package com.moloco.sdk.internal.services

import android.content.Context
import android.net.ConnectivityManager
import android.net.NetworkCapabilities
import android.os.Build
import androidx.core.net.ConnectivityManagerCompat
import com.moloco.sdk.internal.MolocoLogger
import kotlinx.coroutines.delay
import kotlinx.coroutines.withTimeoutOrNull

interface ConnectivityService {
    fun isNetworkMetered(): Boolean

    /**
     * Checks if network is available.
     */
    fun isNetworkAvailable(): Boolean

    /**
     * Waits for network to be available for the given duration.
     * @return true if network is available, false otherwise.
     */
    suspend fun waitForNetwork(duration: Long): Boolean

}

internal class ConnectivityServiceImpl(val context: Context): ConnectivityService {
    private val TAG = "ConnectivityServiceImpl"
    override fun isNetworkMetered(): Boolean {
        return try {
            ConnectivityManagerCompat.isActiveNetworkMetered(context.connectivityManager)
        } catch (e: Exception) {
            MolocoLogger.error("isNetworkMetered", e.toString(), e)
            false
        }
    }

    override suspend fun waitForNetwork(duration: Long): Boolean {
        val hasConnectivity = withTimeoutOrNull(duration) {
            while (!isNetworkAvailable()) {
                MolocoLogger.info(
                    TAG,
                    "waiting because of no network connection"
                )
                delay(100)
            }

            return@withTimeoutOrNull true
        }

        return hasConnectivity ?: false
    }

    @Suppress("DEPRECATION")
    override fun isNetworkAvailable(): Boolean {
        val connectivityManager = context.getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            val network = connectivityManager.activeNetwork ?: return false
            val activeNetwork = connectivityManager.getNetworkCapabilities(network) ?: return false
            return when {
                activeNetwork.hasTransport(NetworkCapabilities.TRANSPORT_WIFI) -> true
                activeNetwork.hasTransport(NetworkCapabilities.TRANSPORT_CELLULAR) -> true
                activeNetwork.hasTransport(NetworkCapabilities.TRANSPORT_ETHERNET) -> true
                else -> false
            }
        } else {
            val networkInfo = connectivityManager.activeNetworkInfo ?: return false
            return networkInfo.isConnected
        }
    }

    private val Context.connectivityManager: ConnectivityManager
        get() = getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager
}

