package com.moloco.sdk.internal.services

import android.content.Context
import android.net.ConnectivityManager
import android.net.ConnectivityManager.RESTRICT_BACKGROUND_STATUS_ENABLED
import android.net.NetworkCapabilities
import android.os.Build
import android.telephony.TelephonyManager
import androidx.annotation.RequiresApi
import com.moloco.sdk.internal.services.NetworkInfo.CellularNetwork
import com.moloco.sdk.internal.services.NetworkInfo.NoNetwork
import com.moloco.sdk.internal.services.NetworkInfo.WifiNetwork

internal interface NetworkInfoService {
    @Deprecated("Use networkInfo()", ReplaceWith("networkInfo()"))
    operator fun invoke(): NetworkInfo

    fun networkInfo(): NetworkInfo

    /**
     * Retrieves the Mobile Network Code (MNC) of the current network.
     * null when MNC is not available (null or empty)
     */
    fun mobileNetworkCode(): Int?

    /**
     * Retrieves the Mobile Country Code (MCC) of the current network.
     * null when MCC is not available (null or empty)
     */
    fun mobileCountryCode(): Int?

    /**
     * Determines if the calling application is subject to metered network restrictions while running on background.
     */
    fun isOnMeteredNetworkRestrictions(): Boolean
}

internal sealed class NetworkInfo {
    object NoNetwork : NetworkInfo()
    object WifiNetwork : NetworkInfo()
    class CellularNetwork(val carrier: String) : NetworkInfo()
}

/**
 * Implementation of [NetworkInfoService] responsible for providing information about the network connectivity.
 *
 * @property context The application context.
 * @property deviceInfoService The service providing device information.
 */
internal class NetworkInfoServiceImpl(
    private val context: Context,
    private val deviceInfoService: DeviceInfoService,
) : NetworkInfoService {

    /**
     * Retrieves network information asynchronously.
     *
     * @return [NetworkInfo] representing the current network state.
     */
    @Deprecated("Use networkInfo()", ReplaceWith("networkInfo()"))
    override fun invoke(): NetworkInfo {
        return networkInfo()
    }

    override fun networkInfo(): NetworkInfo {
        val manager = context.getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager

        return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            manager.networkInfoApi23Plus()
        } else {
            manager.networkInfo()
        }
    }

    override fun mobileNetworkCode(): Int? {
        val telephonyManager = context.getSystemService(Context.TELEPHONY_SERVICE) as TelephonyManager
        val networkOperator = telephonyManager.networkOperator
        return if (!networkOperator.isNullOrEmpty()) {
            networkOperator.substring(3).toInt()
        } else {
            null
        }
    }

    override fun mobileCountryCode(): Int? {
        val telephonyManager = context.getSystemService(Context.TELEPHONY_SERVICE) as TelephonyManager
        val networkOperator = telephonyManager.networkOperator
        return if (!networkOperator.isNullOrEmpty()) {
            networkOperator.substring(0, 3).toInt()
        } else {
            null
        }
    }


    override fun isOnMeteredNetworkRestrictions(): Boolean {
        val connectivityManager = context.getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager
        return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            connectivityManager.networkRestrictedApiNPlus()
        } else {
            false
        }
    }

    @RequiresApi(Build.VERSION_CODES.N)
    private fun ConnectivityManager.networkRestrictedApiNPlus(): Boolean {
        return restrictBackgroundStatus == RESTRICT_BACKGROUND_STATUS_ENABLED
    }

    /**
     * Retrieves network information for devices running API level 23 (Android M) or higher.
     *
     * @return [NetworkInfo] representing the current network state.
     */
    @RequiresApi(Build.VERSION_CODES.M)
    private fun ConnectivityManager.networkInfoApi23Plus(): NetworkInfo {
        if (activeNetwork == null) return NoNetwork
        val networkCapabilities = getNetworkCapabilities(activeNetwork) ?: return NoNetwork

        return when {
            networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_WIFI) -> WifiNetwork
            networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_CELLULAR) -> CellularNetwork(deviceInfoService().mobileCarrier)
            else -> NoNetwork
        }
    }

    /**
     * Retrieves network information for devices running below API level 23 (Android M).
     *
     * @return [NetworkInfo] representing the current network state.
     */
    @Suppress("DEPRECATION")
    private fun ConnectivityManager.networkInfo(): NetworkInfo = when (activeNetworkInfo?.type) {
        ConnectivityManager.TYPE_WIFI -> WifiNetwork
        ConnectivityManager.TYPE_MOBILE -> CellularNetwork(deviceInfoService().mobileCarrier)
        else -> NoNetwork
    }
}

