package com.moloco.sdk.internal.services.bidtoken

import android.os.Build
import androidx.annotation.VisibleForTesting
import com.moloco.sdk.BuildConfig
import com.moloco.sdk.acm.AndroidClientMetrics
import com.moloco.sdk.acm.CountEvent
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.Result
import com.moloco.sdk.internal.bidtoken.BidToken
import com.moloco.sdk.internal.bidtoken.BidTokenParser
import com.moloco.sdk.internal.bidtoken.VersionPrefixedJWTokenParser
import com.moloco.sdk.internal.client_metrics_data.AcmCount
import com.moloco.sdk.internal.client_metrics_data.AcmTag
import com.moloco.sdk.internal.client_metrics_data.AcmTimer
import com.moloco.sdk.internal.http.buildHttpClient
import com.moloco.sdk.internal.services.TimeProviderService
import com.moloco.sdk.internal.services.TimeProviderServiceImpl
import com.moloco.sdk.publisher.Moloco
import com.moloco.sdk.service_locator.SdkObjectFactory
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import java.util.Locale
import java.util.concurrent.TimeUnit
import com.moloco.sdk.acm.AndroidClientMetrics as acm

internal typealias BidTokenFetchResult = BidTokenResponseComponents
internal interface ServerBidTokenService {
    companion object {
        fun create(): ServerBidTokenService = Instance
    }

    suspend fun bidToken(): BidTokenFetchResult
}

/**
 * A stateful service that manages the BidToken at the SDK layer. The BidToken
 * can expire based on numerous factors and this service is responsible for
 * refreshing the token when necessary.
 */
private val Instance by lazy {
    MolocoLogger.info("ServerBidTokenService", "Creating BidTokenService instance")
    ServerBidTokenServiceImpl(
        bidTokenApi = BidTokenApiImpl(
            sdkVersion = BuildConfig.SDK_VERSION_NAME,
            // We create a different http client here because `Moloco` SDK could not be initialized yet and we
            // don't want the factory to cache the http client with null app key httpclient
            httpClient = buildHttpClient(
                appInfo = SdkObjectFactory.DeviceAndApplicationInfo.appInfoSingleton(),
                deviceInfo = SdkObjectFactory.DeviceAndApplicationInfo.deviceInfoSingleton(),
            ),
            httpRequestInfo = BidTokenHttpRequestInfo(
                requestTimeoutMillis = 2800,
                fetchRetryMax = 3,
                fetchRetryDelayMillis = 200, // MAX has a timeout of 3s for the bid token
            ),
            deviceRequestInfo = BidTokenDeviceRequestInfo(
                language = Locale.getDefault().language,
                osVersion = Build.VERSION.RELEASE,
                make = Build.MANUFACTURER ?: "",
                model = Build.MODEL ?: "",
                hardwareVersion = Build.HARDWARE ?: "",
            ),
        ),
        bidTokenParser = VersionPrefixedJWTokenParser(),
        nowUnixMillis = TimeProviderServiceImpl()
    )
}

@VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
internal class ServerBidTokenServiceImpl(
    private val bidTokenApi: BidTokenApi,
    private val bidTokenParser: BidTokenParser,
    private val nowUnixMillis: TimeProviderService
) : ServerBidTokenService {
    private val TAG = "ServerBidTokenServiceImpl"
    private var serverBidToken: BidTokenString = ""
    private var rsaPublicKey: String = ""
    private var bidTokenConfig: BidTokenConfig = DefaultBidTokenConfig

    /**
     * This flag is used to determine if the bid token has been fetched at least once in the same app session
     * This is used for analytics purposes
     */
    private var initialFetch = true

    private val mutex = Mutex()

    private suspend fun shouldRefreshServerBidToken(): Boolean {
        if (serverBidToken.isEmpty()) {
            MolocoLogger.info(TAG, "cached bidToken is empty, needs refresh")
            return true
        }

        when(val bidTokenParseResult = bidTokenParser(serverBidToken)) {
            is Result.Failure -> {
                MolocoLogger.info(TAG, "Failed to parse cached token for expiration, needs refresh")
                return true
            }
            is Result.Success -> {
                val bidToken = bidTokenParseResult.value
                if (bidToken.isExpired(nowUnixMillis)) {
                    MolocoLogger.info(TAG, "Bid token expired, needs refresh")
                    return true
                } else {
                    MolocoLogger.info(TAG, "Bid token has not expired")
                }
            }
        }

        MolocoLogger.info(TAG, "Bid token doesn't need refresh")
        return false
    }

    override suspend fun bidToken(): BidTokenFetchResult {
        // For thread safety
        mutex.withLock {
            if (!shouldRefreshServerBidToken()) {
                MolocoLogger.info(TAG, "bidToken doesn't need refresh, returning cached")
                // initialFetch flag is explicitly always false because we only have in memory caching, so shouldRefreshServerBidToken false means we have 
                // a cached bid token in memory that was previously fetched.
                AndroidClientMetrics.recordCountEvent(CountEvent(AcmCount.ServerBidTokenCached.eventName).withTag(AcmTag.Result.tagName, "true").withTag("initial_fetch", false.toString()))
                return BidTokenFetchResult(serverBidToken, rsaPublicKey, bidTokenConfig)
            }

            val bidTokenTimer = acm.startTimerEvent(AcmTimer.ServerBidTokenFetch.eventName)
            val bidTokenComponents =
                when (val bidTokenFetchResult = bidTokenApi.fetchBidToken()) {
                    is Result.Failure -> {
                        acm.recordCountEvent(CountEvent(AcmCount.ServerBidTokenFetch.eventName)
                            .withTag(AcmTag.Result.tagName, "failure")
                            .withTag(AcmTag.Reason.tagName, "${bidTokenFetchResult.value.failureCode}")
                            .withTag("initial_fetch", initialFetch.toString())
                        )
                        acm.recordTimerEvent(bidTokenTimer
                            .withTag(AcmTag.Result.tagName, "failure")
                            .withTag(AcmTag.Reason.tagName, "${bidTokenFetchResult.value.failureCode}")
                            .withTag("initial_fetch", initialFetch.toString())
                        )
                        MolocoLogger.error(
                            TAG,
                            "bidtoken request failed: ${bidTokenFetchResult.value.failureCode}, details: ${bidTokenFetchResult.value.description}"
                        )
                        DefaultBidTokenResponseComponents
                    }

                    is Result.Success -> {
                        acm.recordCountEvent(CountEvent(AcmCount.ServerBidTokenFetch.eventName)
                            .withTag(AcmTag.Result.tagName, "success")
                            .withTag("initial_fetch", initialFetch.toString())
                        )
                        acm.recordTimerEvent(bidTokenTimer
                            .withTag(AcmTag.Result.tagName, "success")
                            .withTag("initial_fetch", initialFetch.toString())
                        )


                        // This metric is a bit redundant as we should be able to calculate this
                        // from BidTokenFetch - ServerBidTokenFetch (That should give us the # of server bid token calls that were cached)
                        // However to be safe, we are recording this metric here as well.
                        AndroidClientMetrics.recordCountEvent(CountEvent(AcmCount.ServerBidTokenCached.eventName).withTag(AcmTag.Result.tagName, "false").withTag("initial_fetch", initialFetch.toString()))
                        MolocoLogger.info(TAG, "bidtoken request success")
                        bidTokenFetchResult.value.also {
                            // Now that the token has been fetched, we can set the initialFetch flag to false
                            initialFetch = false
                        }
                    }
                }

            this.serverBidToken = bidTokenComponents.bidToken
            this.rsaPublicKey = bidTokenComponents.publicKey
            this.bidTokenConfig = bidTokenComponents.bidTokenConfig

            return bidTokenComponents
        }
    }

    private fun BidToken.isExpired(nowUnixMillis: TimeProviderService): Boolean =
        nowUnixMillis() >= TimeUnit.SECONDS.toMillis(expiresAtUnixSeconds)
}


