package com.moloco.sdk.internal.services.init

import com.moloco.sdk.Init.SDKInitResponse
import com.moloco.sdk.acm.AndroidClientMetrics
import com.moloco.sdk.acm.CountEvent
import com.moloco.sdk.internal.Error
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.Result
import com.moloco.sdk.internal.client_metrics_data.AcmCount
import com.moloco.sdk.internal.client_metrics_data.AcmTag
import com.moloco.sdk.publisher.MediationInfo
import io.ktor.http.HttpStatusCode
import kotlinx.coroutines.delay

internal typealias SDKInitResult = Result<SDKInitResponse, InitFailure>

/**
 * Stateful service that provides capability to invoke init API the SDK as well as cache
 * the last successful init response
 */
internal interface InitService {
    /**
     * @return null if the SDK hasn't been initialized. non-null if SDK has successfully init
     */
    val lastInitResponse: SDKInitResponse?

    /**
     * @param mediationInfo [MediationInfo][com.moloco.sdk.publisher.MediationInfo] required for Moloco SDK, when running in not-direct-integration/bid-token mode.
     * @return Performs init with backend and returns the init response
     */
    suspend fun performInit(
        appKey: String,
        mediationInfo: MediationInfo?
    ): Result<SDKInitResponse, InitFailure>
}

private const val TAG = "InitService"
internal class InitServiceImpl(val initApi: InitApi) : InitService {
    private var _lastInitResponse: SDKInitResponse? = null
    override val lastInitResponse: SDKInitResponse?
        get() = _lastInitResponse

    override suspend fun performInit(
        appKey: String,
        mediationInfo: MediationInfo?
    ): SDKInitResult {
        lateinit var result: Result<SDKInitResponse, InitFailure>

        // Try initializing up to 3 times
        // https://docs.google.com/document/d/1EqQfiptDtcgjwWNET4FuYjQSqUXsxcCbT1JJY67veqA/edit#heading=h.iidd1pqa86oj
        repeat(3) { attempt ->
            result = initApi(appKey, mediationInfo = mediationInfo)

            when (result) {
                is Result.Success -> // If initialization succeeds, return early from the coroutine
                    (result as Result.Success<SDKInitResponse, InitFailure>).value.let {
                        MolocoLogger.info(TAG, "Init, successful in attempt(#$attempt)")
                        _lastInitResponse = it
                        AndroidClientMetrics.recordCountEvent(
                            CountEvent(AcmCount.SDKPerformInitAttempt.eventName).withTag(AcmTag.Result.tagName, "success").withTag(AcmTag.RetryAttempt.tagName, "$attempt")
                        )
                        return@performInit Result.Success(it)
                    }
                is Result.Failure -> (result as Result.Failure<SDKInitResponse, InitFailure>).let {
                    val failureReason = when(val errorType = it.value) {
                        is InitFailure.ClientError -> errorType.type
                        is InitFailure.ServerError -> errorType.statusCode.toString()
                    }
                    AndroidClientMetrics.recordCountEvent(
                        CountEvent(AcmCount.SDKPerformInitAttempt.eventName)
                            .withTag(AcmTag.Result.tagName, "failure")
                            .withTag(AcmTag.RetryAttempt.tagName, "$attempt")
                            .withTag(AcmTag.Reason.tagName, "$failureReason")
                    )
                    MolocoLogger.info(
                        TAG,
                        "Init attempt(#$attempt) failed with error: $failureReason"
                    )

                    // If the error is non-retryable, short-circuit the retry loop
                    if (!it.isRetryable()) {
                        if (it.value is InitFailure.ServerError) {
                            MolocoLogger.error(
                                TAG,
                                "Init response is non-retryable server failure: ${it.value.statusCode}"
                            )
                        } else {
                            MolocoLogger.error(
                                TAG,
                                "Init response is non-retryable server or client failure: ${it.value}"
                            )
                        }
                        return result
                    }
                }
            }

            // Otherwise, wait for 1 second before retrying
            delay(1000)
        }

        MolocoLogger.info(TAG, "Moloco SDK Init failed after all retries: $result")
        return result
    }
}

internal fun Result.Failure<SDKInitResponse, InitFailure>.isRetryable(): Boolean {
    // Only select server errors are retryable
    if (this.value is InitFailure.ServerError) {
        return this.value.statusCode.isRetryable()
    }

    // All client errors are retryable
    return true
}

internal fun Int.isRetryable(): Boolean = this == HttpStatusCode.TooManyRequests.value ||
        this == HttpStatusCode.RequestTimeout.value || this < 400 || this >= 500
