package com.moloco.sdk.publisher

import android.content.Context
import androidx.annotation.VisibleForTesting
import com.moloco.sdk.BuildConfig
import com.moloco.sdk.Init.SDKInitResponse
import com.moloco.sdk.acm.AndroidClientMetrics
import com.moloco.sdk.acm.CountEvent
import com.moloco.sdk.acm.InitConfig
import com.moloco.sdk.acm.AcmHeaders.APP_BUNDLE
import com.moloco.sdk.acm.AcmHeaders.APP_KEY
import com.moloco.sdk.acm.AcmHeaders.APP_VERSION
import com.moloco.sdk.acm.AcmHeaders.MEDIATOR
import com.moloco.sdk.acm.AcmHeaders.MOLOCO_SDK_VERSION
import com.moloco.sdk.acm.AcmHeaders.OS
import com.moloco.sdk.acm.AcmHeaders.OS_VERSION
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.Result
import com.moloco.sdk.internal.android_context.ApplicationContext
import com.moloco.sdk.internal.client_metrics_data.AcmCount
import com.moloco.sdk.internal.client_metrics_data.AcmTag
import com.moloco.sdk.internal.client_metrics_data.AcmTimer
import com.moloco.sdk.internal.configs.DefaultOperationalMetricsConfig
import com.moloco.sdk.internal.configs.OperationalMetricsConfig
import com.moloco.sdk.internal.publisher.AdCreator
import com.moloco.sdk.internal.publisher.AdCreatorConfiguration
import com.moloco.sdk.internal.publisher.InitializationHandler
import com.moloco.sdk.internal.publisher.fireOnUiThread
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.internal.scheduling.GlobalScopes.globalIOScope
import com.moloco.sdk.internal.services.bidtoken.BidTokenHandler
import com.moloco.sdk.internal.services.bidtoken.BidTokenHandlerImpl
import com.moloco.sdk.internal.services.bidtoken.BidTokenService
import com.moloco.sdk.publisher.Moloco.initialize
import com.moloco.sdk.publisher.MolocoAdError.AdCreateError
import com.moloco.sdk.publisher.MolocoAdError.ErrorType
import com.moloco.sdk.publisher.init.MolocoInitParams
import com.moloco.sdk.service_locator.SdkObjectFactory
import com.moloco.sdk.service_locator.SdkObjectFactory.Analytics.applicationLifecycleTrackerSingleton
import com.moloco.sdk.service_locator.SdkObjectFactory.DeviceAndApplicationInfo.appInfoSingleton
import com.moloco.sdk.service_locator.SdkObjectFactory.DeviceAndApplicationInfo.deviceInfoSingleton
import com.moloco.sdk.service_locator.SdkObjectFactory.UserTracking.customUserEventConfigSingleton
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Job
import kotlinx.coroutines.cancelAndJoin
import kotlinx.coroutines.launch
import com.moloco.sdk.acm.AndroidClientMetrics as acm

/**
 * This is an entry point to Moloco SDK.
 * Before creating any ad instances, [initialize] SDK first.
 */
private const val TAG = "Moloco"
object Moloco {
    /**
     * Current initialization status of Moloco SDK.
     * @see initialize
     * @sample com.moloco.sdk.publisher.MolocoIsInitializedSample
     */
    @JvmStatic
    val isInitialized: Boolean
        get() = initializationHandler.initializationState.value==Initialization.SUCCESS

    internal val mediationInfo
        get() = initParams?.mediationInfo
    internal val appKey
        get() = initParams?.appKey

    /**
     * Initializes the Moloco SDK with the given app key and optional initialization listener.
     *
     * If the SDK is already initialized, the provided listener will receive [Initialization.SUCCESS] status.
     *
     * After a successful initialization, you can [createBanner], [createNativeBanner], [createInterstitial], [createRewardedInterstitial].
     *
     * @param initParam The SDK initialization parameters needed to initialize the SDK
     * @param listener an optional listener to receive initialization status updates
     * @see isInitialized
     * @sample com.moloco.sdk.publisher.MolocoInitializeSample
     */
    @JvmStatic
    @JvmOverloads
    @Synchronized
    fun initialize(initParam: MolocoInitParams, listener: MolocoInitializationListener? = null) {
        logMolocoInfo(initParam)
        if (isInitialized) {
            MolocoLogger.info(TAG, "Already initialized. Returning and notifying listener")
            listener?.fireOnUiThread(InitializationHandler.statusAlreadyInitialized)
            return
        }

        if (initJob?.isActive == true) {
            MolocoLogger.info(TAG, "Job active. Returning")
            return
        }

        if (initParam.appKey.isEmpty()) {
            throw IllegalArgumentException("Moloco SDK initialized with empty appKey")
        }

        this.initParams = initParam
        ApplicationContext(initParam.appContext)

        initJob = CoroutineScope(DispatcherProvider().io).launch {
            MolocoLogger.info(TAG, "launched the scope to initialize sdk with thread name: ${Thread.currentThread().name} and dispatcher DispatcherProvider().IO")

            SdkObjectFactory.CrashReporting.crashDetectorSingleton.register()

            val acmInitTimer = acm.startTimerEvent(AcmTimer.SDKInit.eventName)

            // We can calculate sdk failures from SDKInitAttempt - SDKInitSuccess on the dashboard side
            AndroidClientMetrics.recordCountEvent(
                CountEvent(AcmCount.SDKInitAttempt.eventName)
            )
            when (val initResult = initializationHandler.initialize(initParam.appKey, initParam.mediationInfo, SdkObjectFactory.Initialization.trackingApiSingleton)) {
                is Result.Failure -> {
                    // initialization failed, code here can be used for notifying
                    // other APIs if needed

                    MolocoLogger.info(TAG, "Moloco SDK initialization failed")
                    // Callback should be put at the end after all other internal logic has been done
                    listener?.fireOnUiThread(InitializationHandler.statusError(initResult.value.toString()))
                }

                is Result.Success -> {
                    // initialization succeeded, code here can be used for notifying
                    // other APIs if needed
                    processInitConfigs(initResult.value)
                    initializeAndroidClientMetrics(initParam)

                    AndroidClientMetrics.recordCountEvent(
                        CountEvent(AcmCount.SDKInitSuccess.eventName).withTag(AcmTag.Country.tagName, initResult.value.countryIso2Code)
                    )
                    AndroidClientMetrics.recordTimerEvent(acmInitTimer
                        .withTag(AcmTag.Result.tagName, "success")
                        .withTag(AcmTag.Country.tagName, initResult.value.countryIso2Code)
                    )

                    MolocoLogger.info(TAG, "Moloco SDK initialization success")
                    // Callback should be put at the end after all other internal logic has been done
                    listener?.fireOnUiThread(InitializationHandler.statusInitialized)
                }
            }
        }
    }

    private fun logMolocoInfo(initParam: MolocoInitParams) {
        MolocoLogger.info(TAG, "=====================================")
        MolocoLogger.info(TAG, "Moloco SDK initializing")
        MolocoLogger.info(TAG,"SDK Version: ${BuildConfig.SDK_VERSION_NAME}")
        MolocoLogger.info(TAG, "Mediation: ${initParam.mediationInfo.name}")
        MolocoLogger.info(TAG, "isInitialized: $isInitialized")
        MolocoLogger.info(TAG, "=====================================")
    }

    /**
     * Fetches Bid Token asynchronously;
     *
     * _Must be called per each bid request_
     * @param listener use it to get a bid token fetch operation result.
     * @throws Exception when listener is null.
     */
    @JvmStatic
    fun getBidToken(context: Context, listener: MolocoBidTokenListener) {
        ApplicationContext(context)
        MolocoLogger.info(TAG, "[Thread id: ${Thread.currentThread().id}, name: ${Thread.currentThread().name}] Fetching bid token")
        globalIOScope.launch {
            MolocoLogger.info(TAG, "Handling bid token request")
            bidTokenHandler.handleBidTokenRequest(listener)
        }
    }

    /**
     * Moloco Endpoint to pass bid requests to.
     */
    @JvmStatic
    val bidRequestEndpoint: String?
        get() = initializationHandler.response?.adServerUrl?.let {
            if (it.startsWith("http://") || it.startsWith("https://")) {
                return it
            }
            return "https://$it"
        }


    /**
     * Creates an ad object using the provided ad unit ID. Optionally, a watermark string can be added to the ad.
     * The result of the ad creation process is returned through the provided callback, which will either contain
     * the created ad object or an error describing the failure.
     * NOTE - The API requires [Moloco.initialize] to have been called.
     *
     * @param adUnitId The unique identifier for the ad unit to load the ad from. This must be a valid ad unit ID,
     * otherwise, an error of type [AdCreateError.INVALID_AD_UNIT_ID] will be returned.
     * @param watermarkString An optional string that, if provided, will be displayed as a watermark on the ad. If null, no watermark will be applied.
     * @param callback A callback function that will be invoked when the ad creation process is complete. It returns
     * either a valid ad object (e.g., Banner, Interstitial, Rewarded, Native) if successful, or a [AdCreateError]
     * indicating the type of error that occurred.
     * @sample com.moloco.sdk.publisher.MolocoCreateBanner
     */
    @JvmStatic
    fun createBanner(adUnitId: String, watermarkString: String? = null, callback: CreateBannerCallback) {
        MolocoLogger.info(TAG, "[Thread id: ${Thread.currentThread().id}, name: ${Thread.currentThread().name}] Creating banner async for adUnitId: $adUnitId")
        scope.launch {
            val result = adCreator.createBanner(adUnitId, watermarkString)
            val (ad: Banner?, error: AdCreateError?) = when (result) {
                is Result.Success -> result.value to null
                is Result.Failure -> null to result.value
            }

            MolocoLogger.info(TAG, "Banner for adUnitId: $adUnitId has error: ${ad == null}")
            callback(ad, error)
        }
    }

    /**
     * Creates an ad object using the provided ad unit ID. Optionally, a watermark string can be added to the ad.
     * The result of the ad creation process is returned through the provided callback, which will either contain
     * the created ad object or an error describing the failure.
     * NOTE - The API requires [Moloco.initialize] to have been called.
     *
     * @param adUnitId The unique identifier for the ad unit to load the ad from. This must be a valid ad unit ID,
     * otherwise, an error of type [AdCreateError.INVALID_AD_UNIT_ID] will be returned.
     * @param watermarkString An optional string that, if provided, will be displayed as a watermark on the ad. If null, no watermark will be applied.
     * @param callback A callback function that will be invoked when the ad creation process is complete. It returns
     * either a valid ad object (e.g., Banner, Interstitial, Rewarded, Native) if successful, or a [AdCreateError]
     * indicating the type of error that occurred.
     * @sample com.moloco.sdk.publisher.MolocoCreateBannerTablet
     */
    @JvmStatic
    fun createBannerTablet(adUnitId: String, watermarkString: String? = null, callback: CreateBannerCallback) {
        MolocoLogger.info(TAG, "[Thread id: ${Thread.currentThread().id}, name: ${Thread.currentThread().name}] Creating banner tablet async for adUnitId: $adUnitId")
        scope.launch {
            val result = adCreator.createBannerTablet(adUnitId, watermarkString)
            val (ad: Banner?, error: AdCreateError?) = when (result) {
                is Result.Success -> result.value to null
                is Result.Failure -> null to result.value
            }

            MolocoLogger.info(TAG, "Banner for adUnitId: $adUnitId has error: ${ad == null}")
            callback(ad, error)
        }
    }

    @JvmStatic
    @Deprecated("Not supported")
    fun createMREC(adUnitId: String, watermarkString: String? = null, callback: CreateBannerCallback) {
        MolocoLogger.info(TAG, "[Thread id: ${Thread.currentThread().id}, name: ${Thread.currentThread().name}] Creating banner MREC async for adUnitId: $adUnitId")
        scope.launch {
            val result = adCreator.createMREC(adUnitId, watermarkString)
            val (ad: Banner?, error: AdCreateError?) = when (result) {
                is Result.Success -> result.value to null
                is Result.Failure -> null to result.value
            }

            MolocoLogger.info(TAG, "MREC for adUnitId: $adUnitId has error: ${ad == null}")
            callback(ad, error)
        }
    }


    /**
     * Creates an ad object using the provided ad unit ID which provides information about the native ad's assets such as
     * title, description, sponsor text, call to action text, rating, icon URI, main image URI, and video view.
     * All these assets can then be used to stitch together a native ad banner.
     * The result of the ad creation process is returned through the provided callback, which will either contain
     * the created ad object or an error describing the failure.
     * NOTE - The API requires [Moloco.initialize] to have been called.
     *
     * @param adUnitId The unique identifier for the ad unit to load the ad from. This must be a valid ad unit ID,
     * otherwise, an error of type [AdCreateError.INVALID_AD_UNIT_ID] will be returned.
     * @param callback A callback function that will be invoked when the ad creation process is complete. It returns
     * either a valid ad object (e.g., Banner, Interstitial, Rewarded, Native) if successful, or a [AdCreateError]
     * indicating the type of error that occurred.
     * @sample com.moloco.sdk.publisher.MolocoCreateNativeAd
     */
    @JvmStatic
    fun createNativeAd(adUnitId: String, watermarkString: String? = null, callback: CreateNativeAdCallback) {
        MolocoLogger.info(TAG, "[Thread id: ${Thread.currentThread().id}, name: ${Thread.currentThread().name}] Creating native ad for mediation async for adUnitId: $adUnitId")
        scope.launch {
            val result = adCreator.createNativeAd(adUnitId, watermarkString)
            val (ad: NativeAd?, error: AdCreateError?) = when (result) {
                is Result.Success -> result.value to null
                is Result.Failure -> null to result.value
            }
            MolocoLogger.info(TAG, "Native Ad for adUnitId: $adUnitId has error: ${ad == null}")
            callback(ad, error)
        }
    }

    /**
     * Creates an ad object using the provided ad unit ID. Optionally, a watermark string can be added to the ad.
     * The result of the ad creation process is returned through the provided callback, which will either contain
     * the created ad object or an error describing the failure.
     * NOTE - The API requires [Moloco.initialize] to have been called.
     *
     * @param adUnitId The unique identifier for the ad unit to load the ad from. This must be a valid ad unit ID,
     * otherwise, an error of type [AdCreateError.INVALID_AD_UNIT_ID] will be returned.
     * @param watermarkString An optional string that, if provided, will be displayed as a watermark on the ad. If null, no watermark will be applied.
     * @param callback A callback function that will be invoked when the ad creation process is complete. It returns
     * either a valid ad object (e.g., Banner, Interstitial, Rewarded, Native) if successful, or a [AdCreateError]
     * indicating the type of error that occurred.
     * @sample com.moloco.sdk.publisher.MolocoCreateInterstitialAd
     */
    @JvmStatic
    fun createInterstitial(adUnitId: String, watermarkString: String? = null, callback: CreateInterstitialAdCallback) {
        MolocoLogger.info(TAG, "[Thread id: ${Thread.currentThread().id}, name: ${Thread.currentThread().name}] Creating interstitial ad for mediation async for adUnitId: $adUnitId")
        scope.launch {
            val result = adCreator.createInterstitial(adUnitId, watermarkString)
            val (ad: InterstitialAd?, error: AdCreateError?) = when (result) {
                is Result.Success -> result.value to null
                is Result.Failure -> null to result.value
            }

            MolocoLogger.info(TAG, "Interstitial for adUnitId: $adUnitId has error: ${ad == null}")
            callback(ad, error)
        }
    }

    /**
     * Creates an ad object using the provided ad unit ID. Optionally, a watermark string can be added to the ad.
     * The result of the ad creation process is returned through the provided callback, which will either contain
     * the created ad object or an error describing the failure.
     * NOTE - The API requires [Moloco.initialize] to have been called.
     *
     * @param adUnitId The unique identifier for the ad unit to load the ad from. This must be a valid ad unit ID,
     * otherwise, an error of type [AdCreateError.INVALID_AD_UNIT_ID] will be returned.
     * @param watermarkString An optional string that, if provided, will be displayed as a watermark on the ad. If null, no watermark will be applied.
     * @param callback A callback function that will be invoked when the ad creation process is complete. It returns
     * either a valid ad object (e.g., Banner, Interstitial, Rewarded, Native) if successful, or a [AdCreateError]
     * indicating the type of error that occurred.
     * @sample com.moloco.sdk.publisher.MolocoCreateRewardedInterstitialAd
     */
    @JvmStatic
    fun createRewardedInterstitial(adUnitId: String, watermarkString: String? = null, callback: CreateRewardedInterstitialAdCallback) {
        MolocoLogger.info(TAG, "[Thread id: ${Thread.currentThread().id}, name: ${Thread.currentThread().name}] Creating rewarded ad for mediation async for adUnitId: $adUnitId")
        scope.launch {
            val result = adCreator.createRewardedInterstitial(adUnitId, watermarkString)
            val (ad: RewardedInterstitialAd?, error: AdCreateError?) = when (result) {
                is Result.Success -> result.value to null
                is Result.Failure -> null to result.value
            }

            MolocoLogger.info(TAG, "Rewarded for adUnitId: $adUnitId has error: ${ad == null}")
            callback(ad, error)
        }
    }

    private fun processInitConfigs(sdkInitResponse: SDKInitResponse) {
        if (sdkInitResponse.hasEventCollectionConfig()) {
            sdkInitResponse.eventCollectionConfig.apply {
                MolocoLogger.debug(TAG, "Init response has eventCollectionConfig")
                MolocoLogger.debug(TAG, "eventCollectionConfig:")
                MolocoLogger.debug(TAG, "eventCollectionEnabled: $eventCollectionEnabled")
                MolocoLogger.debug(TAG, "mrefCollectionEnabled: $mrefCollectionEnabled")
                MolocoLogger.debug(TAG, "appFgUrl: $appForegroundTrackingUrl")
                MolocoLogger.debug(TAG, "appBgUrl: $appBackgroundTrackingUrl")

                customUserEventConfigSingleton.configure(
                    eventCollectionEnabled = eventCollectionEnabled,
                    userTrackingEnabled = mrefCollectionEnabled,
                    appForegroundUrl = appForegroundTrackingUrl,
                    appBackgroundUrl = appBackgroundTrackingUrl
                )

                if (eventCollectionEnabled) {
                    applicationLifecycleTrackerSingleton.startObserving()
                }
            }
        } else {
            MolocoLogger.debug(TAG, "Init response does not have eventCollectionConfig")
        }
        SdkObjectFactory.Config.configService.initialize(sdkInitResponse)
    }

    private fun initializeAndroidClientMetrics(initParams: MolocoInitParams) {
        val opsConfig = SdkObjectFactory.Config.configService.getConfig(
            configType = OperationalMetricsConfig::class.java,
            default = DefaultOperationalMetricsConfig)
        if (opsConfig.enabled.not()) {
            return
        }

        acm.initialize(InitConfig(
            appId = initParams.appKey,
            postAnalyticsUrl = opsConfig.reportingUrl,
            context = ApplicationContext(),
            requestPeriodSeconds = opsConfig.pollingIntervalSeconds.toLong(),
            mapOf(
                APP_KEY to initParams.appKey,
                APP_BUNDLE to appInfoSingleton().packageName,
                APP_VERSION to appInfoSingleton().version,
                MOLOCO_SDK_VERSION to BuildConfig.SDK_VERSION_NAME,
                OS to deviceInfoSingleton().os,
                OS_VERSION to deviceInfoSingleton().osVersion,
                MEDIATOR to (mediationInfo?.name ?: "")
            )
        )
        )
    }

    @VisibleForTesting(otherwise = VisibleForTesting.NONE)
    internal suspend fun clearState(){
        MolocoLogger.debug(TAG, "clearState() unit testing function called")
        initParams = null
        initJob?.cancelAndJoin()
        initJob = null

        initializationHandler.clearState()
    }

    private val initializationHandler by lazy { InitializationHandler(SdkObjectFactory.Miscellaneous.timeProviderSingleton) }
    private val bidTokenHandler: BidTokenHandler by lazy { BidTokenHandlerImpl(BidTokenService(),  initializationHandler, SdkObjectFactory.Miscellaneous.timeProviderSingleton) }
    private val adCreator by lazy { AdCreator(initializationHandler.initializationState, SdkObjectFactory.Miscellaneous.timeProviderSingleton, AdCreatorConfiguration()) { initializationHandler.awaitAdFactory() } }

    private val scope = CoroutineScope(DispatcherProvider().main)

    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    var initJob: Job? = null
    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    var initParams: MolocoInitParams? = null
}

/**
 * Interface for listening to Moloco initialization status updates.
 * @sample com.moloco.sdk.publisher.MolocoInitializeSample
 */
fun interface MolocoInitializationListener {
    /**
     * Called when there is an update to the Moloco initialization status.
     * @param initStatus the updated initialization status.
     */
    fun onMolocoInitializationStatus(initStatus: MolocoInitStatus)
}

/**
 * Data class representing the Moloco initialization status.
 * @property initialization the [Initialization] status.
 * @property description a description of the initialization status.
 */
data class MolocoInitStatus(val initialization: Initialization, val description: String)

/**
 * Enum class representing the possible Moloco initialization status values.
 */
enum class Initialization { SUCCESS, FAILURE }

/**
 * Moloco bid token listener for [Moloco.getBidToken] call
 */
fun interface MolocoBidTokenListener {
    /**
     * @param bidToken _empty string_ when Moloco SDK is not initialized or in non-bid-token mode; _bid token string_ otherwise.
     * @param error when present it means, [bidToken] is an invalid one: _empty string_. Use [error] data for tracking/logging purposes.
     *
     * _Note: [bidToken] should still be passed to the bid request regardless of [error] presence._
     */
    fun onBidTokenResult(bidToken: String, error: ErrorType?)
}
