package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal

import android.annotation.SuppressLint
import android.content.Context
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.ViewLifecycleOwner
import com.moloco.sdk.internal.ortb.model.Bid
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.service_locator.SdkObjectFactory.AdLoadModule.decLoader
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.AdLoad
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.AdShowListener
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.AggregatedAdShowListener
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.AggregatedOptions
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.CreativeType
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.CreativeType.MRAID
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.CreativeType.STATIC
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.CreativeType.VAST
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.VastAdShowListener
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.Watermark
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.XenossBannerView
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.AdLoadTimeoutError
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.MolocoAdSubErrorType
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.staticrenderer.StaticWebView
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.AdBadgeView
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.VastAdLoader
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import kotlin.time.Duration

/**
 * Banner that resolves ad type (mraid, vast etc) automatically.
 * @param creativeType - bypasses creative resolving part
 * @param adm - ad markup; ad data to load and render.
 * @param dec - double end-card ad data to load and render.
 */
@SuppressLint("ViewConstructor")
internal class AggregatedBanner(
    private val context: Context,
    customUserEventBuilderService: CustomUserEventBuilderService,
    creativeType: CreativeType?,
    private val bid: Bid,
    private val options: AggregatedOptions,
    private val externalLinkHandler: ExternalLinkHandler,
    private val watermark: Watermark,
    private val viewLifecycleOwner: ViewLifecycleOwner,
    private val scope: CoroutineScope
) : XenossBannerView<AggregatedAdShowListener>(context, scope) {
    private val TAG = "AggregatedBanner"
    init {
        tag = "MolocoAggregatedBannerView"
    }

    override var creativeType: CreativeType? = creativeType
        private set

    override var adShowListener: AggregatedAdShowListener? = null
        set(value) {
            field = value

            vastBanner
                ?.let { vastBanner -> vastBanner.adShowListener = value }
                ?: run { (mraidBanner ?: staticBanner)?.adShowListener = value }
        }

    private var vastBanner: XenossBannerView<VastAdShowListener>? = null
    private var mraidBanner: XenossBannerView<AdShowListener>? = null
    private var staticBanner: XenossBannerView<AdShowListener>? = null

    /**
     * Retrieves the currently active banner implementation, prioritizing available banners.
     *
     * This property checks for active banner instances in the following priority:
     * 1. `vastBanner`: A VAST-compliant banner, typically used for video advertisements.
     * 2. `mraidBanner`: An MRAID-compliant banner, suitable for rich media ads.
     * 3. `staticBanner`: A fallback for static banner ads when no dynamic banners are available.
     *
     * If no banners are active, this property returns `null`.
     */
    private val bannerImpl: XenossBannerView<*>?
        get() = vastBanner ?: mraidBanner ?: staticBanner

    override fun destroy() {
        scope.launch {
            super.destroy()
            bannerImpl?.destroy()
        }
    }

    override val adLoader: AdLoad = object : AdLoad {

        private val _isLoaded = MutableStateFlow(false)
        override val isLoaded = _isLoaded.asStateFlow()

        override fun load(timeout: Duration, listener: AdLoad.Listener?) {
            scope.launch {
                prepareBanner()
                // TODO: Banner should never be null at this point. Refactor to make it cleaner
                bannerImpl?.load(timeout, object : AdLoad.Listener {
                    override fun onLoad() {
                        listener?.onLoad()
                    }

                    override fun onLoadTimeout(timeoutError: AdLoadTimeoutError) {
                        when(this@AggregatedBanner.creativeType) {
                            VAST -> {
                                listener?.onLoadTimeout(AdLoadTimeoutError.VAST_BANNER_AD_LOAD_INTERNAL_TIMEOUT_ERROR)
                            }
                            MRAID -> {
                                listener?.onLoadTimeout(AdLoadTimeoutError.MRAID_BANNER_AD_LOAD_INTERNAL_TIMEOUT_ERROR)
                            }
                            STATIC -> {
                                listener?.onLoadTimeout(AdLoadTimeoutError.STATIC_BANNER_AD_LOAD_INTERNAL_TIMEOUT_ERROR)
                            }
                            null ->  MolocoLogger.error(TAG, "creativeType is null", Throwable())
                        }
                    }

                    override fun onLoadError(internalError: MolocoAdSubErrorType) {
                        listener?.onLoadError(internalError)
                    }
                })
            }
        }

        /**
         * Prepares the appropriate banner implementation based on the creative type.
         *
         * This method performs the following:
         * 1. Resolves the `creativeType` if not already available. This may involve performing
         *    potentially heavy operations on a background thread.
         * 2. Initializes the appropriate banner instance (`VastBanner`, `MraidBanner`, or `StaticBanner`)
         *    based on the resolved creative type.
         * 3. Subscribes to the `isLoaded` and `isAdDisplaying` states of the active banner and updates
         *    corresponding LiveData or StateFlows for UI observation.
         * 4. Updates the banner's show listener with the current `adShowListener`.
         */
        // TODO. Duplicate of AggregatedFullscreenAd::prepareAd()
        private suspend fun prepareBanner() {
            MolocoLogger.debug(TAG, "Preparing banner")
            // Possibly heavy stuff.
            val crType = this@AggregatedBanner.creativeType ?: withContext(DispatcherProvider().default) {
                CreativeTypeResolver.resolve(bid.adm).also {
                    this@AggregatedBanner.creativeType = it
                }
            }

            when (crType) {
                VAST -> {
                    /**
                     * VAST relies on Jetpack Compose under the hood for rendering.
                     * To ensure proper lifecycle handling in environments that are not `ComponentActivity`,
                     * a `LifecycleRegistry` must be attached manually.
                     */
                    viewLifecycleOwner.addLifecycleOwnerSupportTo(this@AggregatedBanner)
                    vastBanner = VastBannerView(
                        context = context,
                        customUserEventBuilderService = customUserEventBuilderService,
                        options = options.vastOptions,
                        externalLinkHandler = externalLinkHandler,
                        scope = scope,
                        adLoader = VastAdLoad(
                            bid,
                            scope,
                            VastAdLoader(context),
                            decLoader,
                            false /* No business use case to stream for banners (Figure out?) */
                        )
                    )
                }
                MRAID -> mraidBanner = MraidBannerView(
                    context = context,
                    adm = bid.adm,
                    externalLinkHandler = externalLinkHandler,
                    watermark = watermark,
                    adBadgeView = AdBadgeView(externalLinkHandler, context),
                    scope = scope,
                    )
                STATIC -> {
                    val staticWebView = StaticWebView(
                        context = context,
                        customUserEventBuilderService = customUserEventBuilderService,
                        externalLinkHandler = externalLinkHandler,
                    )
                    staticBanner = StaticBannerView(
                        context = context,
                        watermark = watermark,
                        staticWebView = staticWebView,
                        adBadgeView = AdBadgeView(externalLinkHandler, context),
                        adLoader = StaticAdLoad(bid.adm, scope, staticWebView),
                        scope = scope,
                    )
                }
            }

            bannerImpl?.isLoaded?.onEach {
                _isLoaded.value = it
            }?.launchIn(scope)

            bannerImpl?.isAdDisplaying?.onEach {
                _isAdDisplaying.value = it
            }?.launchIn(scope)

            // Update created banner's show listener.
            adShowListener = adShowListener
        }
    }

    private val _isAdDisplaying = MutableStateFlow(false)
    override val isAdDisplaying = _isAdDisplaying.asStateFlow()

    override fun prepareAdViewForDisplay() {
        adView = bannerImpl
    }
}
