package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal

import android.content.Context
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.ortb.model.Bid
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.service_locator.SdkObjectFactory
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.AdLoad
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.AdShowListener
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.AdWebViewOptions
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.AggregatedAdShowListener
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.AggregatedOptions
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.CreativeType
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.FullscreenAd
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.VastAdShowListener
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.VastOptions
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.Watermark
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.AdLoadTimeoutError
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.AggregatedFullScreenAdError
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.MolocoAdSubErrorType
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.cancel
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import kotlin.time.Duration

/**
 * Class that aggregates each fullscreen creative type and their functionality
 *
 * @param context The application context used for various operations related to the ad.
 * @param customUserEventBuilderService Service used to build custom user events for the ad.
 * @param creativeType The type of creative for the ad. This could be null if not specified.
 * @param bid The bid information associated with the ad.
 * @param externalLinkHandler Handler for processing external links within the ad.
 * @param watermark The watermark to be applied to the ad.
 */
internal class AggregatedFullscreenAd(
    private val context: Context,
    private val customUserEventBuilderService: CustomUserEventBuilderService,
    creativeType: CreativeType?,
    private val bid: Bid,
    private val externalLinkHandler: ExternalLinkHandler,
    private val watermark: Watermark
) : FullscreenAd<AggregatedAdShowListener, AggregatedOptions> {
    private val TAG = "AggregatedFullscreenAd"
    override var creativeType: CreativeType? = creativeType
        private set

    private val scope = CoroutineScope(DispatcherProvider().main)

    private var vastAd: FullscreenAd<VastAdShowListener, VastOptions>? = null
    private var mraidAd: FullscreenAd<AdShowListener, AdWebViewOptions>? = null
    private var staticAd: FullscreenAd<AdShowListener, AdWebViewOptions>? = null
    private val fullscreenAdFactory = FullscreenAdFactoryImpl()

    private val ad: FullscreenAd<*, *>?
        get() = vastAd ?: mraidAd ?: staticAd

    private suspend fun prepareAd() {
        // Possibly heavy stuff.
        val crType = this@AggregatedFullscreenAd.creativeType ?: withContext(DispatcherProvider().default) {
            CreativeTypeResolver.resolve(bid.adm).also {
                this@AggregatedFullscreenAd.creativeType = it
            }
        }

        when (crType) {
            CreativeType.VAST -> vastAd = fullscreenAdFactory.createVastFullscreenAd(context, bid, SdkObjectFactory.Media.mediaConfigSingleton.isStreamingEnabled, watermark)
            CreativeType.MRAID -> mraidAd = fullscreenAdFactory.createMraidFullScreenAd(context, scope, bid, externalLinkHandler, watermark, _isAdDisplaying)
            CreativeType.STATIC -> staticAd = fullscreenAdFactory.createStaticFullscreenAd(context, customUserEventBuilderService, bid.adm, externalLinkHandler, watermark)
        }

        ad?.isLoaded?.onEach {
            _isLoaded.value = it
        }?.launchIn(scope)

        // Copy values from implementation
        ad?.isAdDisplaying?.onEach {
            _isAdDisplaying.value = it
        }?.launchIn(scope)

        // Copy values from implementation
        ad?.isAdForciblyClosed?.onEach {
            _isAdForciblyClosed.value = it
        }?.launchIn(scope)
    }

    private val _isLoaded = MutableStateFlow(false)
    override val isLoaded: StateFlow<Boolean> = _isLoaded

    override fun load(timeout: Duration, listener: AdLoad.Listener?) {
        scope.launch {
            prepareAd()
            // TODO: ad should never be null at this point. Refactor to make it cleaner
            ad?.load(timeout, object : AdLoad.Listener {
                override fun onLoad() {
                    listener?.onLoad()
                }

                override fun onLoadTimeout(timeoutError: AdLoadTimeoutError) {
                    when(this@AggregatedFullscreenAd.creativeType) {
                        CreativeType.VAST -> {
                            listener?.onLoadTimeout(AdLoadTimeoutError.VAST_FULLSCREEN_AD_LOAD_INTERNAL_TIMEOUT_ERROR)
                        }
                        CreativeType.MRAID -> {
                            listener?.onLoadTimeout(AdLoadTimeoutError.MRAID_FULLSCREEN_AD_LOAD_INTERNAL_TIMEOUT_ERROR)
                        }
                        CreativeType.STATIC -> {
                            listener?.onLoadTimeout(AdLoadTimeoutError.STATIC_FULLSCREEN_AD_LOAD_INTERNAL_TIMEOUT_ERROR)
                        }
                        null -> { MolocoLogger.error(TAG, "creativeType is null") }
                    }
                }

                override fun onLoadError(internalError: MolocoAdSubErrorType) {
                    listener?.onLoadError(internalError)
                }
            })
        }
    }

    private val _isAdDisplaying = MutableStateFlow(false)
    override val isAdDisplaying: StateFlow<Boolean> = _isAdDisplaying

    private val _isAdForciblyClosed = MutableStateFlow(false)
    override val isAdForciblyClosed: StateFlow<Boolean> = _isAdForciblyClosed

    override fun show(options: AggregatedOptions, listener: AggregatedAdShowListener?) {
        vastAd?.show(options.vastOptions, listener)
            ?: mraidAd?.show(options.mraidOptions, listener)
            ?: staticAd?.show(options.staticOptions, listener)
            ?: listener?.onShowError(AggregatedFullScreenAdError.FULLSCREEN_AD_SHOW_FAILED_NO_SUPPORTED_TYPE)
    }

    override fun destroy() {
        scope.cancel()
        ad?.destroy()
    }
}
