package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal

import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.channelFlow
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.flow.flow
import kotlinx.coroutines.flow.stateIn

/**
 * @return flow which emits countdown value every second until reaches end.
 *
 * Output example for `countdownSecondsFlow(6u)`:
 *
 * 6s..5s..4s..3s..2s..1s..0s.
 *
 * The last value is always `0` unless the flow is cancelled/failed.
 */
internal fun countdownSecondsFlow(startFromSeconds: UInt): Flow<UInt> = flow {
    var secondsLeft = startFromSeconds

    emit(secondsLeft)

    while (secondsLeft > 0u) {
        delay(1000)
        emit(--secondsLeft)
    }
}

/**
 * @return StateFlow which emits countdown value every second until reaches end.
 *
 * **Countdown value gets updated only when there's at least one active collector;
 * countdown value update pauses automatically when there's no active collectors**
 *
 * Output example for `countdownSecondsFlow(6u)`:
 *
 * 6s..5s..4s..3s..2s..1s..0s.
 *
 * The last value is always `0` unless the flow is cancelled/failed.
 */
internal fun countdownSecondsStateFlow(
    initialSeconds: UInt,
    scope: CoroutineScope
): StateFlow<UInt> {
    var secondsLeft = initialSeconds

    return flow {
        countdownSecondsFlow(secondsLeft).collect {
            secondsLeft = it
            emit(it)
        }
    }.stateIn(scope, SharingStarted.WhileSubscribed(), initialSeconds)
}

/**
 * Same as [countdownSecondsStateFlow()][com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.countdownSecondsStateFlow] but with ability to reset/restart countdown
 *
 * @constructor
 *
 * @param initialSeconds
 */
class ResettableCountdownSecondsStateFlow(
    initialSeconds: UInt,
    private val scope: CoroutineScope
) {

    private val countdownSecondsStateFlows = MutableStateFlow(
        countdownSecondsStateFlow(initialSeconds, scope)
    )

    fun resetTo(initialSeconds: UInt) {
        countdownSecondsStateFlows.value =
            countdownSecondsStateFlow(initialSeconds, scope)
    }

    val countdownSecondsStateFlow: StateFlow<UInt> = channelFlow {
        countdownSecondsStateFlows.collectLatest { sf ->
            sf.collect {
                send(it)
            }
        }
    }.stateIn(scope, SharingStarted.WhileSubscribed(), countdownSecondsStateFlows.value.value)
}
