package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal

import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.error.ErrorMetadata
import com.moloco.sdk.internal.error.ErrorReportingService
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.media.MediaCacheRepository

internal fun interface DECLoader {
    /**
     * Precaches [DEC] object for further rendering; returns the original object if failed to precache as a fallback.
     */
    suspend fun load(decToLoad: DEC, mtid: String?): DEC
}

internal fun DECLoader(mediaCacheRepository: MediaCacheRepository, errorReportingService: ErrorReportingService): DECLoader =
    DECLoaderImpl(mediaCacheRepository, errorReportingService)

internal const val DEC_FAILED_TO_LOAD_ERROR_CODE = "DEC_FAILED_TO_LOAD"
internal const val DEC_LOADED_WITH_NO_APP_ICON_CODE = "DEC_LOADED_WITH_NO_APP_ICON"
private class DECLoaderImpl(
    private val mediaCacheRepository: MediaCacheRepository,
    private val errorReportingService: ErrorReportingService,
) : DECLoader {
    override suspend fun load(decToLoad: DEC, mtid: String?): DEC {
        // No icon to precache, return original object.
        val appIconUri = decToLoad.appIconUri ?: return decToLoad.also {
            errorReportingService.reportError(DEC_LOADED_WITH_NO_APP_ICON_CODE, mtid?.let { ErrorMetadata(it) } ?: ErrorMetadata(mtid = "UNKNOWN_MTID"))
            logD("can't precache DEC: appIconUri is null")
        }

        return when (val precachedIconUriResult = mediaCacheRepository.getMediaFile(appIconUri)) {
            // icon precached, return copy of original DEC object with appIconUri replaced with local-file-based one.
            is MediaCacheRepository.Result.Success -> decToLoad.copy(
                appIconUri = precachedIconUriResult.file.absolutePath
            )
            // return original object if failed to precache.
            else -> null.also {
                errorReportingService.reportError(DEC_FAILED_TO_LOAD_ERROR_CODE, mtid?.let { ErrorMetadata(it) } ?: ErrorMetadata(mtid = "UNKNOWN_MTID"))
                logD("dec loading error: ${precachedIconUriResult}: `Not found` for $appIconUri")
            }
        } ?: decToLoad
    }
}

private const val TAG = "DECLoaderImpl"

private fun logD(s: String) {
    MolocoLogger.debug(TAG, s)
}
