package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal

import com.moloco.sdk.internal.Result
import com.moloco.sdk.internal.ortb.model.Bid
import com.moloco.sdk.internal.toXenossDEC
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.AdLoad
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.AdLoadTimeoutError
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.MraidAdError
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.mraid.MraidAdData
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.TimeoutCancellationException
import kotlinx.coroutines.async
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.launch
import kotlinx.coroutines.withTimeoutOrNull
import kotlin.time.Duration

internal class MraidAdLoad(
    private val scope: CoroutineScope,
    private val bid: Bid?,
    private val decLoader: DECLoader,
    private val loadAndReadyMraid: suspend () -> Result<MraidAdData, MraidAdError>
) : AdLoad {

    private val _isLoaded = MutableStateFlow(false)
    override val isLoaded: StateFlow<Boolean> = _isLoaded
    var mraidLoadResult: Result<MraidAdData, MraidAdError> = Result.Failure(MraidAdError.MRAID_FULLSCREEN_WEBVIEW_CLIENT_UNRECOVERABLE_ERROR)

    override fun load(timeout: Duration, listener: AdLoad.Listener?) {
        scope.launch {

            if (mraidLoadResult is Result.Success) {
                listener?.onLoad()
                return@launch
            }

            val mraidLoadResultDeferred = async {
                withTimeoutOrNull(timeout) {
                    loadAndReadyMraid()
                }
            }

            val dec = bid?.ext?.player?.dec?.toXenossDEC()

            // This logic is purely around precached dec
            val decDeferred = async {
                withTimeoutOrNull(timeout) {
                    dec?.let {
                        try {
                            decLoader.load(it, bid?.ext?.mtid)
                        } catch (e: Exception) {
                            dec
                        }
                    }
                } ?: dec // fallback in case timed out; still use original dec object.
            }

            val adLoadResult = try {
                mraidLoadResultDeferred.await()
            } catch (e: TimeoutCancellationException) {
                // if main ad didn't load in time, there's no need to wait for DEC.
                decDeferred.cancel()
                // Notify ad load timeout.
                val timeoutError = AdLoadTimeoutError.MRAID_FULLSCREEN_AD_LOAD_INTERNAL_TIMEOUT_ERROR
                mraidLoadResult = Result.Failure(MraidAdError.MRAID_WEBVIEW_INTERNAL_TIMEOUT_ERROR)
                listener?.onLoadTimeout(timeoutError)
                return@launch
            }

            when (adLoadResult) {
                null -> listener?.onLoadTimeout(AdLoadTimeoutError.MRAID_AD_LOAD_INTERNAL_TIMEOUT_ERROR)

                is Result.Failure -> {
                    listener?.onLoadError(adLoadResult.value)
                    decDeferred.cancel()
                }

                is Result.Success -> {
                    mraidLoadResult = Result.Success(adLoadResult.value.copy(dec = decDeferred.await()))
                    _isLoaded.value = true
                    listener?.onLoad()
                }
            }
        }
    }
}
