package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal

import android.annotation.SuppressLint
import android.content.Context
import android.view.Gravity
import androidx.annotation.VisibleForTesting
import com.moloco.sdk.internal.utils.launchUnit
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.AdShowListener
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.CreativeType
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.Watermark
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.XenossBannerView
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.toStaticBannerError
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.staticrenderer.StaticWebView
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.AdBadgeView
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.DEFAULT_PRIVACY_URL
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.dpToPx
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.launch

@SuppressLint("ViewConstructor")
internal class StaticBannerView(
    private val context: Context,
    @get:VisibleForTesting
    val watermark: Watermark,
    private val staticWebView: StaticWebView,
    private val adBadgeView: AdBadgeView,
    override val adLoader: StaticAdLoad,
    private val scope: CoroutineScope,
) : XenossBannerView<AdShowListener>(context, scope) {
    init {
        tag = "MolocoStaticBannerView"
    }

    override val creativeType = CreativeType.STATIC

    override fun destroy() {
        super.destroy()
        staticWebView.destroy()
    }

    override fun prepareAdViewForDisplay() = scope.launchUnit {
        launch {
            val error = staticWebView.unrecoverableError.first { it != null }
            error?.let {
                adShowListener?.onShowError(it.toStaticBannerError())
            }
        }
        launch {
            staticWebView.clickthroughEvent.collect {
                adShowListener?.onClick()
            }
        }

        adBadgeView.apply {
            layoutParams = LayoutParams(
                LayoutParams.WRAP_CONTENT,
                LayoutParams.WRAP_CONTENT
            ).apply {
                gravity = Gravity.BOTTOM or Gravity.START
                setPadding(AD_BADGE_PADDING.dpToPx(context), 0, 0, AD_BADGE_PADDING.dpToPx(context))
                setPrivacyUrl(DEFAULT_PRIVACY_URL)
                setOnButtonRenderedListener { staticWebView::onButtonRendered }
            }
        }

            // TODO: https://mlc.atlassian.net/browse/SDK-3311
            adView = wrapWebView(context, staticWebView, adBadgeView).also {
                watermark.applyWatermark(it)
            }
    }

    override val isAdDisplaying: StateFlow<Boolean> by lazy {
        // Ensure accurate tracking for static ads by combining WebView's "isPageFinished" event
        // with the parent isAdDisplaying state. This prevents discrepancies between notifnt
        // and HTML-embedded trackers.
        super.isAdDisplaying
            .combine(staticWebView.isPageFinished) { superIsAdDisplaying, isPageFinished ->
                superIsAdDisplaying && isPageFinished
            }
            .stateIn(
                scope = scope,
                started = SharingStarted.Eagerly,
                initialValue = false
            )
    }

    companion object {
        /**
         * Indicates the padding at which the Ad Badge should be rendered in the bottom left corner (the i icon)
         */
        private const val AD_BADGE_PADDING = 5
    }
}
