package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui

import android.annotation.SuppressLint
import android.content.Context
import android.util.AttributeSet
import android.widget.FrameLayout
import android.widget.ImageButton
import android.widget.ImageView
import androidx.annotation.VisibleForTesting
import com.moloco.sdk.R
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ExternalLinkHandler
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService


/**
 * A custom view that displays an ad badge button (the "i" icon) in the bottom left corner of the parent layout in a non-composable context.
 * The button allows users to access a privacy URL when clicked and notifies our button trackers when it is rendered.
 *
 * @constructor Creates an instance of [AdBadgeView].
 * @param context The context of the view.
 * @param attrs Optional attribute set to customize the view.
 * @param defStyleAttr An optional style resource to define default styling for the view.
 */
@SuppressLint("ViewConstructor")
internal class AdBadgeView @JvmOverloads constructor(
    private val externalLinkHandler: ExternalLinkHandler,
    context: Context,
    attrs: AttributeSet? = null,
    defStyleAttr: Int = 0
) : FrameLayout(context, attrs, defStyleAttr) {

    /**
     * The size of the Ad Badge in dp
     */
    private val AD_BADGE_SIZE = 12

    /**
     * The `ImageButton` representing the Ad Badge.
     */
    private val adButton: ImageButton = ImageButton(context).apply {
        layoutParams = LayoutParams(
            AD_BADGE_SIZE.dpToPx(context),
            AD_BADGE_SIZE.dpToPx(context)
        )
        setImageResource(R.drawable.info_badge)
        scaleType = ImageView.ScaleType.CENTER_INSIDE
        clipToOutline = true
        background = null

        setOnClickListener {
            externalLinkHandler(DEFAULT_PRIVACY_URL)
        }
    }

    /**
     * A listener invoked when the button is rendered, providing information about its position and size.
     */
    private var onButtonRenderedListener: ((CustomUserEventBuilderService.UserInteraction.Button) -> Unit)? = null

    init {
        addView(adButton)
    }

    /**
     * Provide the global position and size of the Ad Badge button.
     *
     * @param changed Indicates whether the layout dimensions have changed.
     * @param left The left position of this view within its parent.
     * @param top The top position of this view within its parent.
     * @param right The right position of this view within its parent.
     * @param bottom The bottom position of this view within its parent.
     */
    @SuppressLint("DrawAllocation")
    override fun onLayout(changed: Boolean, left: Int, top: Int, right: Int, bottom: Int) {
        super.onLayout(changed, left, top, right, bottom)

        if (changed) {
            val density = resources.displayMetrics.density

            val locationOnScreen = IntArray(2)
            adButton.getLocationOnScreen(locationOnScreen)
            val globalLeft = locationOnScreen[0].toFloat() / density
            val globalTop = locationOnScreen[1].toFloat() / density
            val widthDp = adButton.width.toFloat() / density
            val heightDp = adButton.height.toFloat() / density

            val button = CustomUserEventBuilderService.UserInteraction.Button(
                buttonType = CustomUserEventBuilderService.UserInteraction.Button.ButtonType.AD_BADGE,
                position = CustomUserEventBuilderService.UserInteraction.Position(
                    topLeftXDp = globalLeft,
                    topLeftYDp = globalTop
                ),
                size = CustomUserEventBuilderService.UserInteraction.Size(
                    widthDp = widthDp,
                    heightDp = heightDp
                )
            )

            onButtonRenderedListener?.invoke(button)
        }
    }

    /**
     * Sets a listener to be called when the Ad Badge button is rendered.
     * The listener receives the button's global position and size information.
     *
     * @param listener A lambda function to handle button rendering events.
     */
    fun setOnButtonRenderedListener(listener: (CustomUserEventBuilderService.UserInteraction.Button) -> Unit) {
        onButtonRenderedListener = listener
    }

    /**
     * Sets the URL to be opened when the Ad Badge button is clicked.
     *
     * @param url The new privacy URL.
     */
    fun setPrivacyUrl(url: String) {
        adButton.setOnClickListener {
            externalLinkHandler(url)
        }
    }

    @VisibleForTesting
    fun getAdButton(): ImageButton {
        return adButton
    }
}

/**
 * Extension function to convert a DP (density-independent pixel) value to pixels.
 *
 * @receiver The value in DP to be converted.
 * @param context The context providing display metrics for conversion.
 * @return The equivalent pixel value.
 */
fun Int.dpToPx(context: Context): Int {
    return (this * context.resources.displayMetrics.density).toInt()
}
