package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui

import androidx.compose.animation.Crossfade
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.size
import androidx.compose.material.ButtonDefaults
import androidx.compose.material.MaterialTheme
import androidx.compose.material.Text
import androidx.compose.material.TextButton
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.Shape
import androidx.compose.ui.graphics.painter.Painter
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.unit.DpSize
import androidx.compose.ui.unit.TextUnit
import com.moloco.sdk.internal.ortb.model.CountDownTimer
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.rewardedcountdowntimer.AggregateRewardedCountDownTimer
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.utils.DEFAULT_BUTTON_DP_SIZE
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.utils.DEFAULT_FONT_SIZE

// Stateless view for "3..2..1.. then button appears" UI functionality support.
// Button part can be either "icon" or "text". "Enabled" does not apply to the "countdown" view part.
@Composable
internal fun CountdownButton(
    countdownButtonPart: CountdownButtonPart,
    modifier: Modifier = Modifier,
    onClick: () -> Unit,
    enabled: Boolean,
    color: Color = MaterialTheme.colors.primary,
    size: DpSize = DEFAULT_BUTTON_DP_SIZE,
    fontSize: TextUnit = DEFAULT_FONT_SIZE,
    countDownTimer: CountDownTimer?,
) {
    Box(
        modifier = Modifier
            .fillMaxSize()
            .size(size),
        contentAlignment = Alignment.TopEnd
    ) {

        if (countdownButtonPart is CountdownButtonPart.Countdown
            && countDownTimer != null
            && countdownButtonPart.enableCountDownTimer
        ) {
            AggregateRewardedCountDownTimer(
                countDownTimer = countDownTimer,
                totalCountDownTime = countdownButtonPart.totalCountdownTime,
                currentCountDownTime = countdownButtonPart.currentCountdownTime
            )
        }

        Crossfade(countdownButtonPart, modifier) { buttonPart ->
            val textStyle = MaterialTheme.typography.h6

            when (buttonPart) {
                is CountdownButtonPart.Countdown -> {
                    // Due to flicker from CrossFade we remove the countdown timer rendering inside this section and place it above.
                    // However, we still want the IconButton to fade in so we keep this logic in place for the other button parts
                }
                is CountdownButtonPart.IconButton -> {
                    GenericIconButton(
                        painter = buttonPart.painter,
                        onClick = onClick,
                        enabled = enabled,
                        tint = color,
                        size = size,
                        iconSize = buttonPart.iconSize,
                        backgroundShape = buttonPart.backgroundShape,
                        backgroundColor = buttonPart.backgroundColor,
                        contentDescription = buttonPart.contentDescription
                    )
                }
                is CountdownButtonPart.TextButton -> TextButton(
                    onClick = onClick,
                    enabled = enabled,
                    colors = ButtonDefaults.textButtonColors(
                        contentColor = color,
                        disabledContentColor = color
                    )
                ) {
                    Text(
                        text = buttonPart.text,
                        style = textStyle,
                        maxLines = 1,
                        fontSize = fontSize,
                        textAlign = TextAlign.Center
                    )
                }
            }
        }
    }
}

sealed class CountdownButtonPart {
    class Countdown(val enableCountDownTimer: Boolean, val totalCountdownTime: UInt, val currentCountdownTime: UInt) : CountdownButtonPart()

    class IconButton(
        val painter: Painter,
        val contentDescription: String?,
        val iconSize: DpSize,
        val backgroundShape: Shape,
        val backgroundColor: Color
    ) : CountdownButtonPart()

    class TextButton(val text: String) : CountdownButtonPart()
}
