package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render

import android.content.Context
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.Destroyable
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ExternalLinkHandler
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.VastResource
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.launch

interface VastPrivacyIcon : Destroyable {
    val vastPrivacyIcon: StateFlow<PreparedVastResource?>
    fun onVastPrivacyIconClick()
    fun onVastPrivacyIconDisplayed()
}

internal fun VastPrivacyIcon(
    icon: VastResource?,
    widthPx: Int?,
    heightPx: Int?,
    clickLink: String?,
    scope: CoroutineScope,
    context: Context,
    customUserEventBuilderService: CustomUserEventBuilderService,
    externalLinkHandler: ExternalLinkHandler,
    trackClick: (() -> Unit)?,
    trackDisplay: (() -> Unit)?
): VastPrivacyIcon = VastPrivacyIconImpl(
    icon,
    widthPx,
    heightPx,
    clickLink,
    scope,
    context,
    customUserEventBuilderService,
    externalLinkHandler,
    trackClick,
    trackDisplay
)

private class VastPrivacyIconImpl(
    private val icon: VastResource?,
    widthPx: Int?,
    heightPx: Int?,
    private val clickLink: String?,
    scope: CoroutineScope,
    context: Context,
    private val customUserEventBuilderService: CustomUserEventBuilderService,
    private val externalLinkHandler: ExternalLinkHandler,
    private val trackClick: (() -> Unit)?,
    private val trackDisplay: (() -> Unit)?
) : VastPrivacyIcon {

    private val iconResourceHandler = MutableStateFlow<PreparedVastResourceHandler?>(null)

    init {
        // Preload icon resource immediately.
        scope.launch {
            var preparedVastResourceHandler: PreparedVastResourceHandler? = null

            try {
                preparedVastResourceHandler = icon?.prepareVastResource(
                    context,
                    customUserEventBuilderService,
                    externalLinkHandler,
                    widthPx ?: 0,
                    heightPx ?: 0,
                    onWebViewClick = {},
                    onWebViewError = {}
                )

                iconResourceHandler.value = preparedVastResourceHandler
            } catch (e: Exception) {
                // Releasing resources in case of any error.
                preparedVastResourceHandler?.destroy()
                destroy()
            }
        }
    }

    override fun destroy() {
        with(iconResourceHandler) {
            value?.destroy()
            value = null
        }
    }

    override val vastPrivacyIcon: StateFlow<PreparedVastResource?> = iconResourceHandler.map {
        it?.resource
    }.stateIn(scope, SharingStarted.WhileSubscribed(), null)

    override fun onVastPrivacyIconClick() {
        clickLink?.let {
            trackClick?.invoke()
            externalLinkHandler(it)
        }
    }

    override fun onVastPrivacyIconDisplayed() {
        trackDisplay?.invoke()
    }
}
