package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.companion

import android.content.Context
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ExternalLinkHandler
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.MolocoAdSubErrorType
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.utils.ScreenUtils.toPosition
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.Companion
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.PreparedVastResource
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.PreparedVastResourceHandler
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.prepareVastResource
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService.UserInteraction.Button
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService.UserInteraction.Button.ButtonType
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService.UserInteraction.Position
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.cancel
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.launch

internal class CompanionControllerImpl(
    private val companion: Companion,
    goNextActionDelaySeconds: UInt,
    private val context: Context,
    private val customUserEventBuilderService: CustomUserEventBuilderService,
    private val externalLinkHandler: ExternalLinkHandler
) : CompanionController {

    private val scope = CoroutineScope(DispatcherProvider().main)

    private val goNextActionHolder = CompanionGoNextAction(
        goNextActionDelaySeconds,
        scope
    )

    override val goNextAction by goNextActionHolder::goNextAction

    override fun reset() = goNextActionHolder.reset()

    private var lastClickPosition = androidx.compose.ui.geometry.Offset.Zero.toPosition()

    private val tracker = CompanionVastTracker(
        customUserEventBuilderService,
        companion.clickTracking,
        companion.creativeViewTracking
    )

    private val _event = MutableSharedFlow<CompanionControllerEvent>()
    override val event: Flow<CompanionControllerEvent> = _event

    private fun onEvent(event: CompanionControllerEvent) = scope.launch { _event.emit(event) }

    override fun onError(error: MolocoAdSubErrorType) {
        onEvent(CompanionControllerEvent.Error(error))
    }

    override fun onButtonRendered(button: Button) {
        tracker.trackButtonRender(button)
    }

    override fun onButtonUnRendered(buttonType: ButtonType) {
        tracker.trackButtonUnRender(buttonType)
    }

    override fun onDisplayStarted() {
        tracker.trackCreativeView()
        onEvent(CompanionControllerEvent.DisplayStarted)
    }

    override fun onLastClickPosition(position: Position) {
        lastClickPosition = position
    }

    // TODO. Currently only called by image endcard; static/mraid cards have mind of their own.
    override fun onClickThrough(position: Position) {
        companion.clickThroughUrl?.let {
            tracker.trackClick(position)
            externalLinkHandler(it)
            onEvent(CompanionControllerEvent.ClickThrough)
        }
    }

    override val hasClickThrough: Boolean = companion.clickThroughUrl != null

    private var resourceHandler: PreparedVastResourceHandler? = null
        set(value) {
            field = value
            _resource.value = value?.resource
        }

    private val _resource = MutableStateFlow(resourceHandler?.resource)
    override val resource: StateFlow<PreparedVastResource?> = _resource

    init {
        scope.launch {
            resourceHandler = companion.resource.prepareVastResource(
                context,
                customUserEventBuilderService,
                externalLinkHandler,
                companion.widthPx,
                companion.heightPx,
                // TODO. Proper solution. Ugly.
                // companion.clicktrhough doesn't play a role here, since
                // onClickThrough() is called only by an image companion, but in case of static/mraid companions
                // we always consider them as ad playables, therefore, we don't have a click event interception;
                // instead we wait for them to redirect the user to the different page which is the substitution for companion.clicktrhough link.
                onWebViewClick = {
                    tracker.trackClick(lastClickPosition)
                    onEvent(CompanionControllerEvent.ClickThrough)
                },
                onWebViewError = { error ->
                    // TODO. https://mlc.atlassian.net/browse/SDK-1728
                    //  Pass error: MolocoAdSubErrorType here.
                    this@CompanionControllerImpl.onError(error)
                }
            )
        }
    }

    override fun destroy() {
        scope.cancel()
        resourceHandler?.destroy()
        resourceHandler = null
    }
}
