package com.moloco.sdk.internal

import android.content.Context
import com.moloco.sdk.Init.SDKInitResponse
import com.moloco.sdk.Init.SDKInitResponse.AdUnit.InventoryType
import com.moloco.sdk.acm.AndroidClientMetrics
import com.moloco.sdk.internal.ortb.BidResponseParser
import com.moloco.sdk.internal.publisher.AdCreateLoadTimeoutManager
import com.moloco.sdk.internal.publisher.BidLoader
import com.moloco.sdk.internal.publisher.InterstitialAd
import com.moloco.sdk.internal.publisher.RewardedInterstitialAd
import com.moloco.sdk.internal.publisher.bidPreprocessors
import com.moloco.sdk.internal.publisher.nativead.NativeAdAssetsProvider
import com.moloco.sdk.internal.publisher.nativead.NativeAdImpl
import com.moloco.sdk.internal.publisher.nativead.NativeAdLoader
import com.moloco.sdk.internal.publisher.nativead.parser.NativeAdOrtbResponseParser
import com.moloco.sdk.internal.services.AnalyticsApplicationLifecycleTracker
import com.moloco.sdk.internal.services.AudioService
import com.moloco.sdk.internal.services.ClickthroughService
import com.moloco.sdk.internal.services.TimeProviderService
import com.moloco.sdk.publisher.Banner
import com.moloco.sdk.publisher.InterstitialAd
import com.moloco.sdk.publisher.NativeAd
import com.moloco.sdk.publisher.RewardedInterstitialAd
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.Watermark
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ExternalLinkHandler
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ViewVisibilityTracker
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService
import com.moloco.sdk.xenoss.sdkdevkit.android.persistenttransport.PersistentHttpRequest

internal interface AdFactory {
    fun createBanner(
            context: Context,
            appLifecycleTrackerService: AnalyticsApplicationLifecycleTracker,
            adUnitId: String,
            viewVisibilityTracker: ViewVisibilityTracker,
            externalLinkHandler: ExternalLinkHandler,
            watermark: Watermark,
            adCreateLoadTimeoutManager: AdCreateLoadTimeoutManager,
            viewLifecycleOwnerSingleton: ViewLifecycleOwner,
            bannerSize: BannerSize
    ): Banner?

    fun createBannerTablet(
            context: Context,
            appLifecycleTrackerService: AnalyticsApplicationLifecycleTracker,
            adUnitId: String,
            viewVisibilityTracker: ViewVisibilityTracker,
            externalLinkHandler: ExternalLinkHandler,
            watermark: Watermark,
            adCreateLoadTimeoutManager: AdCreateLoadTimeoutManager,
            viewLifecycleOwnerSingleton: ViewLifecycleOwner,
            bannerSize: BannerSize
    ): Banner?

    fun createMREC(
            context: Context,
            appLifecycleTrackerService: AnalyticsApplicationLifecycleTracker,
            adUnitId: String,
            viewVisibilityTracker: ViewVisibilityTracker,
            externalLinkHandler: ExternalLinkHandler,
            watermark: Watermark,
            adCreateLoadTimeoutManager: AdCreateLoadTimeoutManager,
            viewLifecycleOwnerSingleton: ViewLifecycleOwner,
            bannerSize: BannerSize
    ): Banner?

    fun createInterstitial(
            context: Context,
            appLifecycleTrackerService: AnalyticsApplicationLifecycleTracker,
            adUnitId: String,
            viewVisibilityTracker: ViewVisibilityTracker,
            externalLinkHandler: ExternalLinkHandler,
            persistentHttpRequest: PersistentHttpRequest,
            watermark: Watermark,
            adCreateLoadTimeoutManager: AdCreateLoadTimeoutManager
    ): InterstitialAd?

    fun createRewardedInterstitial(
            context: Context,
            appLifecycleTrackerService: AnalyticsApplicationLifecycleTracker,
            adUnitId: String,
            viewVisibilityTracker: ViewVisibilityTracker,
            externalLinkHandler: ExternalLinkHandler,
            persistentHttpRequest: PersistentHttpRequest,
            watermark: Watermark,
            adCreateLoadTimeoutManager: AdCreateLoadTimeoutManager
    ): RewardedInterstitialAd?

    fun createNativeAd(
        context: Context,
        appLifecycleTrackerService: AnalyticsApplicationLifecycleTracker,
        audioService: AudioService,
        adUnitId: String,
        viewVisibilityTracker: ViewVisibilityTracker,
        externalLinkHandler: ExternalLinkHandler,
        persistentHttpRequest: PersistentHttpRequest,
        viewLifecycleOwnerSingleton: ViewLifecycleOwner,
        watermark: Watermark,
        adCreateLoadTimeoutManager: AdCreateLoadTimeoutManager,
        timeProvider: TimeProviderService,
    ): NativeAd?
}

internal class AdFactoryImpl(
    private val initResponse: SDKInitResponse,
    private val customUserEventBuilderService: CustomUserEventBuilderService,
    private val clickthroughService: ClickthroughService
) : AdFactory {
    private val verifyBannerVisible by lazy { initResponse.verifyBannerVisible }

    private val adUnits: Map<InventoryType, Set<String>> by lazy {
        initResponse.let { initResponse ->
            val adUnits = listOf(
                InventoryType.BANNER,
                InventoryType.INTERSTITIAL,
                InventoryType.REWARD_VIDEO,
                InventoryType.NATIVE
            ).associateWith {
                mutableSetOf(
                    // For enabling test mode. https://mlc.atlassian.net/browse/SDK-175
                    MOLOCO_TEST_AD_UNIT_ID,
                    MOLOCO_MAX_OFFICIAL_BANNER_TEST_AD_UNIT_ID,
                    MOLOCO_MAX_OFFICIAL_INTERSTITIAL_TEST_AD_UNIT_ID,
                    MOLOCO_MAX_OFFICIAL_REWARDED_TEST_AD_UNIT_ID,

                    LEVEL_PLAY_OFFICIAL_TEST_BANNER_AD_UNIT_ID,
                    LEVEL_PLAY_OFFICIAL_TEST_INTERSTITIAL_AD_UNIT_ID,
                    LEVEL_PLAY_OFFICIAL_TEST_REWARDED_AD_UNIT_ID
                )
            }

            initResponse.adUnitsList.onEach {
                adUnits[it.type]?.add(it.id)
            }

            adUnits
        }
    }

    private fun canCreateAd(type: InventoryType, adUnitId: String): Boolean =
        adUnits[type]?.contains(adUnitId) == true

    override fun createBanner(
            context: Context,
            appLifecycleTrackerService: AnalyticsApplicationLifecycleTracker,
            adUnitId: String,
            viewVisibilityTracker: ViewVisibilityTracker,
            externalLinkHandler: ExternalLinkHandler,
            watermark: Watermark,
            adCreateLoadTimeoutManager: AdCreateLoadTimeoutManager,
            viewLifecycleOwnerSingleton: ViewLifecycleOwner,
            bannerSize: BannerSize
    ): Banner? = if (canCreateAd(InventoryType.BANNER, adUnitId)) {
        com.moloco.sdk.internal.publisher.BannerView(
            context,
            appLifecycleTrackerService,
            customUserEventBuilderService,
            adUnitId,
            verifyBannerVisible,
            externalLinkHandler,
            watermark,
            adCreateLoadTimeoutManager,
            viewLifecycleOwnerSingleton,
            bannerSize,
            clickthroughService
        )
    } else {
        null
    }

    override fun createBannerTablet(
            context: Context,
            appLifecycleTrackerService: AnalyticsApplicationLifecycleTracker,
            adUnitId: String,
            viewVisibilityTracker: ViewVisibilityTracker,
            externalLinkHandler: ExternalLinkHandler,
            watermark: Watermark,
            adCreateLoadTimeoutManager: AdCreateLoadTimeoutManager,
            viewLifecycleOwnerSingleton: ViewLifecycleOwner,
            bannerSize: BannerSize
    ): Banner? = if (canCreateAd(InventoryType.BANNER, adUnitId)) {
        com.moloco.sdk.internal.publisher.BannerView(
            context,
            appLifecycleTrackerService,
            customUserEventBuilderService,
            adUnitId,
            verifyBannerVisible,
            externalLinkHandler,
            watermark,
            adCreateLoadTimeoutManager,
            viewLifecycleOwnerSingleton,
            bannerSize,
            clickthroughService
        )
    } else {
        null
    }

    override fun createMREC(
            context: Context,
            appLifecycleTrackerService: AnalyticsApplicationLifecycleTracker,
            adUnitId: String,
            viewVisibilityTracker: ViewVisibilityTracker,
            externalLinkHandler: ExternalLinkHandler,
            watermark: Watermark,
            adCreateLoadTimeoutManager: AdCreateLoadTimeoutManager,
            viewLifecycleOwnerSingleton: ViewLifecycleOwner,
            bannerSize: BannerSize
    ): Banner? = if (canCreateAd(InventoryType.BANNER, adUnitId)) {
        com.moloco.sdk.internal.publisher.BannerView(
            context,
            appLifecycleTrackerService,
            customUserEventBuilderService,
            adUnitId,
            verifyBannerVisible,
            externalLinkHandler,
            watermark,
            adCreateLoadTimeoutManager,
            viewLifecycleOwnerSingleton,
            bannerSize,
            clickthroughService
        )
    } else {
        null
    }

    override fun createNativeAd(
        context: Context,
        appLifecycleTrackerService: AnalyticsApplicationLifecycleTracker,
        audioService: AudioService,
        adUnitId: String,
        viewVisibilityTracker: ViewVisibilityTracker,
        externalLinkHandler: ExternalLinkHandler,
        persistentHttpRequest: PersistentHttpRequest,
        viewLifecycleOwnerSingleton: ViewLifecycleOwner,
        watermark: Watermark,
        adCreateLoadTimeoutManager: AdCreateLoadTimeoutManager,
        timeProvider: TimeProviderService,
    ): NativeAd? = if (canCreateAd(InventoryType.NATIVE, adUnitId)) {
        val nativeAdLoader = NativeAdLoader(
            context,
            adUnitId,
            BidLoader(BidResponseParser(), bidPreprocessors()),
            NativeAdOrtbResponseParser(),
            adCreateLoadTimeoutManager,
            AndroidClientMetrics,
            timeProvider)

        val nativeAdAssetsProvider = NativeAdAssetsProvider(
            context,
            externalLinkHandler,
            audioService.isMediaVolumeOnMute,
            customUserEventBuilderService,
            viewVisibilityTracker,
            viewLifecycleOwnerSingleton,
            watermark
        )

        NativeAdImpl(
            adUnitId,
            nativeAdLoader,
            nativeAdAssetsProvider,
            appLifecycleTrackerService,
            customUserEventBuilderService,
            externalLinkHandler,
            persistentHttpRequest,
            adCreateLoadTimeoutManager,
        )
    } else {
        null
    }

    override fun createInterstitial(
            context: Context,
            appLifecycleTrackerService: AnalyticsApplicationLifecycleTracker,
            adUnitId: String,
            viewVisibilityTracker: ViewVisibilityTracker,
            externalLinkHandler: ExternalLinkHandler,
            persistentHttpRequest: PersistentHttpRequest,
            watermark: Watermark,
            adCreateLoadTimeoutManager: AdCreateLoadTimeoutManager,
    ): InterstitialAd? = if (canCreateAd(InventoryType.INTERSTITIAL, adUnitId)) {
        InterstitialAd(
                context,
                appLifecycleTrackerService,
                customUserEventBuilderService,
                adUnitId,
                externalLinkHandler = externalLinkHandler,
                persistentHttpRequest = persistentHttpRequest,
                watermark = watermark,
                adCreateLoadTimeoutManager = adCreateLoadTimeoutManager,
        )
    } else {
        null
    }

    override fun createRewardedInterstitial(
            context: Context,
            appLifecycleTrackerService: AnalyticsApplicationLifecycleTracker,
            adUnitId: String,
            viewVisibilityTracker: ViewVisibilityTracker,
            externalLinkHandler: ExternalLinkHandler,
            persistentHttpRequest: PersistentHttpRequest,
            watermark: Watermark,
            adCreateLoadTimeoutManager: AdCreateLoadTimeoutManager
    ): RewardedInterstitialAd? = if (canCreateAd(InventoryType.REWARD_VIDEO, adUnitId)) {
        RewardedInterstitialAd(
                context,
                appLifecycleTrackerService,
                customUserEventBuilderService,
                adUnitId,
                externalLinkHandler = externalLinkHandler,
                persistentHttpRequest = persistentHttpRequest,
                watermark = watermark,
                adCreateLoadTimeoutManager = adCreateLoadTimeoutManager,
          )
    } else {
        null
    }
}

/**
 * The constant string representing the test ad unit id
 */
const val MOLOCO_TEST_AD_UNIT_ID = "moloco_test_placement"

/*
 * MAX Test Ad Unit Ids
 */
private const val MOLOCO_MAX_OFFICIAL_BANNER_TEST_AD_UNIT_ID = "m8Ue4CTEIiSfJQEA"
private const val MOLOCO_MAX_OFFICIAL_INTERSTITIAL_TEST_AD_UNIT_ID = "Ratv4sDzSI5hSEku"
private const val MOLOCO_MAX_OFFICIAL_REWARDED_TEST_AD_UNIT_ID = "mfnJ3YIVB4eCmhQD"
private const val MOLOCO_MAX_OFFICIAL_NATIVE_VIDEO_TEST_AD_UNIT_ID = "PdHKCrJsOy3qVIIr"
private const val MOLOCO_MAX_OFFICIAL_NATIVE_IMAGE_TEST_AD_UNIT_ID = "cZQSJpHegsQdLQGP"
private const val MOLOCO_MAX_OFFICIAL_NATIVE_LOGO_TEST_AD_UNIT_ID = "eDpyjrZ1BZxisS1r"

/*
 * LevelPlay Test Ad Unit Ids
 */
private const val LEVEL_PLAY_OFFICIAL_TEST_BANNER_AD_UNIT_ID = "DvTjZQ9VR1mYQGJM"
private const val LEVEL_PLAY_OFFICIAL_TEST_INTERSTITIAL_AD_UNIT_ID = "DLZ8sDK5OpsKC7Hv"
private const val LEVEL_PLAY_OFFICIAL_TEST_REWARDED_AD_UNIT_ID = "N3y1oKosmyXSEkyZ"
