package com.moloco.sdk.internal

import android.content.res.Resources

/**
 * Represents the dimensions of a banner ad in density-independent pixels (dp).
 *
 * @property wDp The width of the banner in dp.
 * @property hDp The height of the banner in dp.
 */
internal sealed class BannerSize(open val wDp: Int, open val hDp: Int) {
    /**
     * Standard banner size
     *
     * - Width: 320dp
     * - Height: 50dp
     */
    object Standard : BannerSize(320, 50)

    /**
     * Banner size optimized for tablets.
     *
     * - Width: 728dp
     * - Height: 90dp
     */
    object Tablet : BannerSize(728, 90)

    /**
     * Medium Rectangle (MREC) banner size.
     *
     * - Width: 300dp
     * - Height: 250dp
     */
    object MREC : BannerSize(300, 250)

    /**
     * Custom banner size defined by the user.
     *
     * @param wDp The width of the banner in dp.
     * @param hDp The height of the banner in dp.
     */
    data class Custom(override val wDp: Int, override val hDp: Int) : BannerSize(wDp, hDp)
}


/**
 * Converts a value in density-independent pixels (dp) to pixels (px) based on the device's screen density. For use in Layout parameters
 *
 * @receiver The value in dp.
 * @return The equivalent value in pixels.
 */
internal fun Int.toPx(): Int = (this * Resources.getSystem().displayMetrics.density).toInt()
