package com.moloco.sdk.internal

import androidx.compose.animation.AnimatedVisibility
import androidx.compose.animation.animateContentSize
import androidx.compose.foundation.background
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.displayCutoutPadding
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.layout.widthIn
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.semantics.Role
import androidx.compose.ui.text.font.Font
import androidx.compose.ui.text.font.FontFamily
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.unit.dp
import coil.compose.AsyncImage
import com.moloco.sdk.R
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.TrackableButton
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.utils.DEFAULT_AD_BUTTON_PADDING
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.ad.AdViewModel
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.compose.CTAButton
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService.UserInteraction.Button.ButtonType

// Based on xenoss.sdkdevkit.android.defaultCTAButton() composable.
@Composable
internal fun molocoCTAButton(
    alignment: Alignment = Alignment.BottomEnd,
    padding: PaddingValues = PaddingValues(DEFAULT_AD_BUTTON_PADDING),
    text: String = stringResource(id = R.string.com_moloco_sdk_xenoss_player_learn_more),
    contentColor: Color = Color.White,
    backgroundColor: Color = MolocoBlue,
    imageUri: String? = null
): CTAButton = { ctaAvailable, currentAdPartFlow, onButtonRendered, onCTA ->
    val currentAdPart by currentAdPartFlow.collectAsState()
    AnimatedVisibility(
        visible = ctaAvailable,
        modifier = Modifier
            .align(alignment)
            .displayCutoutPadding()
            .padding(padding)
    ) {
        // We show the same Vast CTA between linear and companion, however
        // because linear and companion have different view models, they have different
        // states of which buttons are shown. The ButtonType.CTA is de-duped at jetpack compose
        // to avoid repeatedly getting notification on a recompose but no changes to
        // the position/sizing. The side effect is that when we go from Linear -> Companion
        // we have no way to trigger a button rendered callback to the companion view model.
        // Thus we trigger a recomposition here by rendering a different instance of the
        // composable. This is not ideal but due to how JP compose spams us with button modifiers
        // we need to de-dupe to avoid clobbering the main thread and thus to workaround the de-dupe
        // we need to re-render one additional time. Visually this has no impact
        when (currentAdPart) {
            is AdViewModel.AdPart.Companion -> {
                TrackableButton(
                    buttonType = ButtonType.CTA,
                    onButtonRendered = onButtonRendered
                ) {
                    MolocoVastCTA(
                        modifier = it,
                        onClick = onCTA,
                        text = text,
                        contentColor = contentColor,
                        backgroundColor = backgroundColor,
                        imageUri = imageUri
                    )
                }
            }
            is AdViewModel.AdPart.Linear -> {
                TrackableButton(
                    buttonType = ButtonType.CTA,
                    onButtonRendered = onButtonRendered
                ) {
                    MolocoVastCTA(
                        modifier = it,
                        onClick = onCTA,
                        text = text,
                        contentColor = contentColor,
                        backgroundColor = backgroundColor,
                        imageUri = imageUri
                    )
                }
            }
            is AdViewModel.AdPart.DEC -> {/* no-op */ }
            is AdViewModel.AdPart.Mraid -> {/* no-op */ }
            null -> {/* no-op */}
        }
    }
}

// Based on xenoss.sdkdevkit.android.VastCTA() composable.
@Composable
private fun MolocoVastCTA(
    modifier: Modifier = Modifier,
    imageUri: String? = null,
    text: String,
    contentColor: Color,
    backgroundColor: Color,
    onClick: () -> Unit
) {
    val shape = RoundedCornerShape(4.dp)

    Row(
        modifier = modifier
            .widthIn(max = 164.dp)
            .clip(shape)
            .background(backgroundColor)
            .animateContentSize()
            .clickable(onClick = onClick, role = Role.Button, onClickLabel = "CTA"),
        horizontalArrangement = Arrangement.Center,
        verticalAlignment = Alignment.CenterVertically
    ) {
        imageUri?.let {
            Spacer(modifier = Modifier.width(4.dp))

            AsyncImage(
                modifier = Modifier
                    .size(36.dp)
                    .clip(shape),
                model = imageUri,
                contentDescription = null
            )
        }

        Text(
            modifier = Modifier.padding(
                horizontal = 15.dp,
                vertical = 12.dp
            ),
            color = contentColor,
            text = text,
            maxLines = 1,
            fontFamily = FontFamily.Default,
            fontWeight = FontWeight.Bold,
        )
    }
}

internal val MolocoBlue = Color(0xFF0280FB)
