package com.moloco.sdk.internal.mediators

import kotlin.time.Duration
import kotlin.time.DurationUnit
import kotlin.time.toDuration

// Mediators are case sensitive
internal enum class Mediators {
    MAX,
    LevelPlay,
    X3MADS,
    AdMob,
    Custom_MAX,
    Custom_LevelPlay,
    Custom_AdMob,
}

/**
 * Mediators that can process base64 gzipped bidresponse
 */
internal fun canProcessBase64GzippedBidResponse(mediator: String?): Boolean {
    return mediator == Mediators.MAX.name || mediator == Mediators.LevelPlay.name || mediator == Mediators.AdMob.name
}

/**
 * Calculates the timeout duration for loading an interstitial ad for the given mediator.
 *
 * @param mediator The mediator for which the timeout duration is to be calculated. Can be null.
 * @return The duration for the ad load timeout.
 */
// TODO: Make this configurable - https://mlc.atlassian.net/browse/SDK-1784
internal fun fullScreenAdLoadTimeout(mediator: String?): Duration {
    return when (mediator) {
        /**
         * For those wondering how we came to this number...
         * It took planning and guile. Some sleepless nights and the moto, try hard or die trying.
         * Finally we arrived at the conclusion:
         * Applovin MAX has a 30 seconds timeout. We subtract 4 second to avoid timeout race condition.
         */
        Mediators.MAX.name -> 26.toDuration(DurationUnit.SECONDS)

        // AdMob has a timeout of 30-45 seconds. We go with the most aggressive timeout.
        Mediators.AdMob.name -> 26.toDuration(DurationUnit.SECONDS)

        // Unity LevelPlay has a 27 second timeout. We subtract 1 second to avoid a timeout race condition.
        Mediators.LevelPlay.name -> 26.toDuration(DurationUnit.SECONDS)

        // Applovin MAX custom adapter MAX has a 15 seconds timeout. We subtract 1 second to avoid timeout race condition.
        // That's what we use for every other custom adapter as well
        else -> 14.toDuration(DurationUnit.SECONDS)
    }
}

/**
 * Calculates the timeout duration for loading a banner ad for the given mediator.
 *
 * @param mediator The mediator for which the timeout duration is to be calculated. Can be null.
 * @return The duration for the ad load timeout.
 */
internal fun bannerAdLoadTimeout(mediator: String?): Duration {
    return when (mediator) {
        /**
         * For those wondering how we came to this number...
         * It took planning and guile. Some sleepless nights and the moto, try hard or die trying.
         * Finally we arrived at the conclusion:
         * Applovin MAX has a 10 seconds timeout. We subtract 2 second to avoid timeout race condition.
         * Unity LevelPlay a 10 seconds timeout. We subtract 2 second to avoid timeout race condition.
         */
        Mediators.MAX.name -> 8.toDuration(DurationUnit.SECONDS)
        Mediators.LevelPlay.name -> 8.toDuration(DurationUnit.SECONDS)
        Mediators.AdMob.name -> 8.toDuration(DurationUnit.SECONDS)
        else -> 9.toDuration(DurationUnit.SECONDS)
    }
}

/**
 * Calculates the timeout duration for loading a banner ad for the given mediator.
 *
 * @param mediator The mediator for which the timeout duration is to be calculated. Can be null.
 * @return The duration for the ad load timeout.
 */
internal fun nativeAdLoadTimeout(mediator: String?): Duration {
    return when (mediator) {
        /**
         * Max uses 10 seconds for timeout. We subtract 2 second to avoid timeout race condition.
         */
        Mediators.MAX.name -> 8.toDuration(DurationUnit.SECONDS)
        /**
         * Admob uses 60 seconds for timeout. We subtract 2 second to avoid timeout race condition.
         */
        Mediators.AdMob.name -> 58.toDuration(DurationUnit.SECONDS)
        else -> 8.toDuration(DurationUnit.SECONDS)
    }
}
