package com.moloco.sdk.internal.publisher

import androidx.annotation.VisibleForTesting
import com.moloco.sdk.Init
import com.moloco.sdk.internal.AdFactory
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.Result
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.internal.scheduling.runOnMainDispatcher
import com.moloco.sdk.internal.services.TimeProviderService
import com.moloco.sdk.internal.services.init.ClientFailureType
import com.moloco.sdk.internal.services.init.InitFailure
import com.moloco.sdk.internal.services.init.InitTrackingApi
import com.moloco.sdk.publisher.Initialization
import com.moloco.sdk.publisher.MediationInfo
import com.moloco.sdk.publisher.MolocoInitStatus
import com.moloco.sdk.publisher.MolocoInitializationListener
import com.moloco.sdk.service_locator.SdkObjectFactory
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.withContext

/**
 * Handles the initialization process for the Moloco SDK, ensuring that the SDK is initialized
 * correctly with the provided application key and mediation information. This class manages the
 * initialization flow, handles potential errors, and provides mechanisms for checking the
 * initialization status and retrieving the `AdFactory` once the SDK is initialized.
 *
 * This class is internal and is not meant to be accessed directly by external components.
 */
internal class InitializationHandler(val timeProviderService: TimeProviderService) {
    private val _initializationState = MutableStateFlow<Initialization?>(null)

    /**
     * null - SDK initialization has not yet been completed
     * success - SDK initialization has completed with success
     * failure - SDK initialization has completed with failure.
     */
    val initializationState = _initializationState.asStateFlow()

    /**
     * Holds the SDK initialization response, if available. This property is set when the SDK
     * initialization succeeds and contains the details of the initialization result.
     *
     * This property is nullable and will be `null` if the SDK initialization has not been
     * performed yet or if it failed. The value of this property can only be modified internally,
     * and it is read-only from outside the class.
     *
     * @see initialize
     */
    var response: Init.SDKInitResponse? = null
        private set

    /**
     * This method is used to check if the SDK can be initialized or not. If this condition fails, then calling initialize() will result in init failure
     */
    internal fun canInitialize(): Boolean {
        // https://mlc.atlassian.net/browse/SDK-1061
        // We are getting a WorkManager crash that prevents us from creating a persistent
        // http request client. Thus we report that error to our tracking service and
        // cause the Moloco SDK to not initialize

        return isPersistentHttpRequestAvailable()
    }

    /**
     * Suspends until [initialize] completes successfully and returns the initialized [AdFactory].
     *
     * This function listens to the [_adFactoryFlow], which emits [AdFactory] instances upon successful
     * initialization of the Moloco SDK. It returns the first distinct [AdFactory] instance that is emitted
     * after initialization is complete.
     *
     * @return The `AdFactory` instance emitted after the successful initialization of the Moloco SDK.
     */
    suspend fun awaitAdFactory(): AdFactory {
        MolocoLogger.info(TAG, "Moloco SDK awaiting init to receive AdFactory")
        val adFactory = _adFactoryFlow.first { it != null } as AdFactory
        MolocoLogger.info(TAG, "Moloco SDK init completed, AdFactory received")
        return adFactory
    }

    /**
     * Initializes the SDK with the provided `appKey` and `mediationInfo`. This method handles the
     * initialization process.
     *
     * @param appKey The application key required to initialize the SDK.
     * @param mediationInfo The mediation information needed for SDK initialization.
     *
     */
    suspend fun initialize(
            appKey: String,
            mediationInfo: MediationInfo,
            trackingApi: InitTrackingApi,
    ): Result<Init.SDKInitResponse, InitFailure> {
        MolocoLogger.info(TAG, "initialize()")

        if (canInitialize().not()) {
            MolocoLogger.error(TAG, "PersistentHttpRequest is not available, failing to initialize")
            val clientFailureType = InitFailure.ClientError(ClientFailureType.PersistentHttpUnavailableError)
            trackingApi.notifyFailure(clientFailureType, 0L)
            _initializationState.emit(Initialization.FAILURE)
            return Result.Failure(clientFailureType)
        }

        return startInitialization(appKey, mediationInfo, trackingApi)
    }

    private suspend fun startInitialization(
            appKey: String,
            mediationInfo: MediationInfo,
            trackingApi: InitTrackingApi,
    ) = withContext(DispatcherProvider().io) {
        MolocoLogger.info(TAG, "startInitialization switch to Dispatchers.IO")

        // Perform the initialization
        val startTime = timeProviderService.currentTime()
        val result = SdkObjectFactory.Initialization.initSingleton.performInit(appKey, mediationInfo)
        val latency = timeProviderService.currentTime() - startTime

        // Handle the initialization result
        handleInitializationResult(result, latency, trackingApi)

        // Return the initialization result
        result
    }

    private suspend fun handleInitializationResult(result: Result<Init.SDKInitResponse, InitFailure>, latency: Long, trackingApi: InitTrackingApi) = when (result) {
        is Result.Failure -> {
            _initializationState.emit(Initialization.FAILURE)
            MolocoLogger.info(TAG, "sdk init failed")
            // If initialization failed, call `initTrackingApi` with a failure result
            trackingApi.notifyFailure(result.value, latency)
        }

        is Result.Success -> {
            _initializationState.emit(Initialization.SUCCESS)
            MolocoLogger.info(TAG, "sdk init success")
            with(result.value) {
                response = this
                trackingApi.notifySuccess(latency = latency)
                _adFactoryFlow.emit(SdkObjectFactory.Initialization.adFactorySingleton(this))
            }
        }
    }

    private fun isPersistentHttpRequestAvailable() = try {
        SdkObjectFactory.Network.persistentHttpRequestSingleton
        true
    } catch (e: IllegalStateException) {
        false
    }

    @VisibleForTesting(otherwise = VisibleForTesting.NONE)
    internal fun clearState() {
        response = null
        _adFactoryFlow.value = null
        _initializationState.value = null
    }

    private val _adFactoryFlow = MutableStateFlow<AdFactory?>(null)

    companion object {
        /**
         * A pre-initialized [MolocoInitStatus] object representing already initialized state.
         */
        val statusAlreadyInitialized =
                MolocoInitStatus(Initialization.SUCCESS, "Already Initialized")

        /**
         * Returns a new [MolocoInitStatus] object representing initialization error.
         * @param errorMessage a description of the error that occurred.
         */
        fun statusError(errorMessage: String) =
                MolocoInitStatus(Initialization.FAILURE, errorMessage)

        /**
         * A pre-initialized [MolocoInitStatus] object representing successful initialization.
         */
        val statusInitialized = MolocoInitStatus(Initialization.SUCCESS, "Initialized")

        private const val TAG = "InitializationHandler"
    }
}

fun MolocoInitializationListener.fireOnUiThread(initStatus: MolocoInitStatus) {
    runOnMainDispatcher { onMolocoInitializationStatus(initStatus) }
}
