package com.moloco.sdk.internal.publisher.nativead

import android.content.Context
import android.net.Uri
import android.view.View
import com.moloco.sdk.internal.ViewLifecycleOwner
import com.moloco.sdk.internal.publisher.nativead.model.NativeAdAssetIds
import com.moloco.sdk.internal.publisher.nativead.model.PreparedNativeAssets
import com.moloco.sdk.internal.publisher.nativead.ui.NativeAdImageContainer
import com.moloco.sdk.internal.publisher.nativead.ui.VideoContainer
import com.moloco.sdk.publisher.NativeAd
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.Watermark
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ExternalLinkHandler
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ViewVisibilityTracker
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.Ad
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.ad.AdController
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.ad.vastAdPlaylistController
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService

internal class NativeAdAssetsProvider(
    private val context: Context,
    private val externalLinkHandler: ExternalLinkHandler,
    private val isMediaVolumeOnMute: Boolean,
    private val customUserEventBuilderService: CustomUserEventBuilderService,
    private val viewVisibilityTracker: ViewVisibilityTracker,
    private val viewLifecycleOwner: ViewLifecycleOwner,
    private val watermark: Watermark
) : NativeAd.Assets {
    var onClick: (() -> Unit)? = null

    /**
     * Assets only get prepared once [NativeAdImpl.createAdLoadCallback] finishes processing Assets.
     */
    var preparedAssets: PreparedNativeAssets? = null

    override val title: String?
        get() = preparedAssets?.getTitlesFrom(NativeAdAssetIds.TITLE)

    override val description: String?
        get() = preparedAssets?.getDataFrom(NativeAdAssetIds.DESCRIPTION)

    override val sponsorText: String?
        get() = preparedAssets?.getDataFrom(NativeAdAssetIds.SPONSOR_TEXT)

    override val callToActionText: String?
        get() = preparedAssets?.getDataFrom(NativeAdAssetIds.CTA_TEXT)

    override val rating: Float?
        get() = preparedAssets?.getDataFrom(NativeAdAssetIds.RATING)?.toFloatOrNull()

    override val iconUri: Uri?
        get() = preparedAssets?.getImageUriFor(NativeAdAssetIds.ICON)

    override val mediaView: View?
        get() {
            val vastAd = preparedAssets?.getVastAdFor(NativeAdAssetIds.VIDEO)
            val imageUri = preparedAssets?.getImageUriFor(NativeAdAssetIds.MAIN_IMAGE)

            return when {
                vastAd != null -> createVideoView(vastAd)
                imageUri != null -> createImageView(imageUri)
                else -> null
            }
        }

    override val mainImageUri: Uri? = preparedAssets?.getImageUriFor(NativeAdAssetIds.MAIN_IMAGE)


    private var image: View? = null
    private var vastAdController: AdController? = null
    private var video: VideoContainer? = null

    /**
     * Creates a video view for the VAST ad, reusing a cached instance if available.
     *
     * This method checks if a video view has already been created and cached. If so, it returns the cached view.
     * Otherwise, it initializes a [vastAdPlaylistController] for handling the VAST ad playback,
     * and uses it to create a new [VideoContainer] instance. The newly created view is cached for future use.
     *
     * @param vastAd The VAST ad object containing the data required for video playback.
     * @return A [View] instance representing the video container for the VAST ad.
     */
    private fun createVideoView(vastAd: Ad): View {
        video?.let { return it }

        val vastAdController = vastAdPlaylistController(
            ad = vastAd,
            externalLinkHandler = externalLinkHandler,
            context = context,
            mute = isMediaVolumeOnMute,
            overrideLinearGoNextActionEnabled = false,
            overrideLinearGoNextActionEnabledDelaySeconds = 0,
            companionGoNextActionDelaySeconds = 0,
            decGoNextActionDelaySeconds = 0,
            autoStoreOnComplete = false,
            autoStoreOnSkip = false,
            customUserEventBuilderService = customUserEventBuilderService
        ).also {
            it.show()
            vastAdController = it
        }

        return VideoContainer(context, vastAdController, viewVisibilityTracker, viewLifecycleOwner, watermark, onClick).also {
            video = it
        }
    }

    /**
     * Creates an image view for the native ad, reusing a cached instance if available.
     *
     * @param imageUri The URI of the image to display in the native ad.
     * @return A [View] instance representing the native ad image.
     */
    private fun createImageView(imageUri: Uri): View{
        image?.let { return it }

        return NativeAdImageContainer(
            context = context,
            imageUri = imageUri,
            watermark = watermark,
            onClick = onClick
        ).also { image = it }
    }

    fun destroy() {
        vastAdController?.destroy()
        vastAdController = null

        video?.destroyUnderlyingVideoComposeView()
        video = null
    }
}
