package com.moloco.sdk.internal.publisher.nativead

import com.moloco.sdk.internal.ortb.model.Bid
import com.moloco.sdk.internal.publisher.BUrlData
import com.moloco.sdk.internal.publisher.InternalAdShowListener
import com.moloco.sdk.internal.publisher.InternalAdShowListenerTracker
import com.moloco.sdk.internal.publisher.nativead.model.NativeOrtbResponse
import com.moloco.sdk.internal.services.AnalyticsApplicationLifecycleTracker
import com.moloco.sdk.publisher.AdFormatType
import com.moloco.sdk.publisher.createAdInfo
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ExternalLinkHandler
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService
import com.moloco.sdk.xenoss.sdkdevkit.android.persistenttransport.PersistentHttpRequest

/**
 * A listener class responsible for managing the tracking of impressions and clicks for native ads.
 * It handles event reporting by integrating with internal trackers and external services.
 *
 * This class combines general ad show tracking functionality with native ad-specific tracking, ensuring
 * proper reporting of ad visibility (impressions) and user interactions (clicks).
 *
 * ## Responsibilities:
 * - Tracks and reports impressions using URLs provided in the native ad response.
 * - Tracks and reports clicks by invoking external link handlers and sending click tracking URLs.
 * - Integrates with application lifecycle analytics and custom event builder services for enhanced reporting.
 *
 * ## Dependencies:
 * - [adUnitId]: Identifier of the ad unit for which the listener is configured.
 * - [bid]: The winning bid data containing pricing and URL information for tracking purposes.
 * - [ortbResponse]: The response object from the OpenRTB native ad request, including tracking data.
 * - [appLifecycleTrackerService]: Service for tracking app lifecycle events for analytics.
 * - [customUserEventBuilderService]: Service for building custom user analytics events.
 * - [adFormatType]: Type of ad format (e.g., native) for differentiation in tracking.
 * - [persistentHttpRequest]: Utility for sending HTTP requests for tracking URLs.
 * - [externalLinkHandler]: Handler for opening external links associated with the ad.
 *
 * ## Internal Components:
 * - [generalAdShowTracker]: Tracks generic ad show events and integrates with application-level analytics.
 * - [nativeAdSpecificTracker]: Handles tracking specific to native ads, including impressions and clicks.
 *
 * ## Public Methods:
 * - [fireImpressionListenerAndTracker]: Triggers impression tracking and notifies the general ad show tracker.
 * - [fireClickListenerAndTracker]: Triggers click tracking, opens the associated link, and notifies the general tracker.
 */
internal class NativeAdShowListenerWithTracker(
    private val adUnitId: String,
    private val bid: Bid,
    private val ortbResponse: NativeOrtbResponse,
    private val appLifecycleTrackerService: AnalyticsApplicationLifecycleTracker,
    private val customUserEventBuilderService: CustomUserEventBuilderService,
    private val adFormatType: AdFormatType,
    private val persistentHttpRequest: PersistentHttpRequest,
    private val externalLinkHandler: ExternalLinkHandler,
) {
    private val generalAdShowTracker: InternalAdShowListener = createAdShowTracker()
    private val nativeAdSpecificTracker: NativeAdSpecificTrackers = createNativeAdSpecificTracker()

    fun fireImpressionListenerAndTracker() {
        nativeAdSpecificTracker.trackImpressions()
        generalAdShowTracker.onAdShowSuccess(createAdInfo(adUnitId))
    }

    fun fireClickListenerAndTracker() {
        ortbResponse.link?.let {
            nativeAdSpecificTracker.trackLink(it.clickTrackerUrls)
            externalLinkHandler.invoke(it.url)
        }

        generalAdShowTracker.onAdClicked(createAdInfo(adUnitId))
    }

    private fun createAdShowTracker() = with(bid) {
        InternalAdShowListenerTracker(
            originListener = null, // Native ads handle show listener themselves, due to the way they work
            appLifecycleTrackerService = appLifecycleTrackerService,
            customUserEventBuilderService = customUserEventBuilderService,
            provideSdkEvents = { ext.sdkEvents }, // TODO: once the rest of the AdFormats use the new AdLoader make this a non lambda
            provideBUrlData = { bid.burl?.let { BUrlData(it, bid.price) } }, // TODO: once the rest of the AdFormats use the new AdLoader make this a non lambda
            adType = adFormatType
        )
    }

    private fun createNativeAdSpecificTracker() = with(ortbResponse) {
        NativeAdSpecificTrackers(
            impressionTrackerUrls,
            eventTrackers,
            persistentHttpRequest,
        )
    }

    /**
     * Trackers specific to native ads
     */
    private class NativeAdSpecificTrackers(
        private var impressionTrackerUrls: List<String>?,
        private var eventTrackers: List<NativeOrtbResponse.EventTracker>?,
        private val persistentHttpRequest: PersistentHttpRequest,
    ) {
        fun trackImpressions() {
            impressionTrackerUrls?.forEach {
                persistentHttpRequest.send(it)
            }
            impressionTrackerUrls = null

            eventTrackers?.forEach { event ->
                if (event.url != null &&
                    event.eventType == NativeOrtbEventTrackerEventType.IMPRESSION &&
                    event.methodType == NativeOrtbEventTrackerMethodType.IMAGE
                ) {
                    persistentHttpRequest.send(event.url)
                }
            }
            eventTrackers = null
        }

        private val trackedLinkUrls = mutableSetOf<String>()

        fun trackLink(urls: List<String>) {
            for (url in urls) {
                if (!trackedLinkUrls.contains(url)) {
                    persistentHttpRequest.send(url)
                    trackedLinkUrls += url
                }
            }
        }

        object NativeOrtbEventTrackerEventType {
            const val IMPRESSION = 1
        }

        object NativeOrtbEventTrackerMethodType {
            const val IMAGE = 1
        }
    }
}
