package com.moloco.sdk.internal.services

import androidx.annotation.MainThread
import androidx.lifecycle.DefaultLifecycleObserver
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleOwner
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.internal.services.analytics.AnalyticsService
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch

/**
 * Tracks the foreground and background of the application and
 * emits Project Telepathy analytics
 */
internal interface AnalyticsApplicationLifecycleTracker {

    /**
     * Allows the lifecycle tracker to start observing application lifecycle events.
     * To capture the events in analytics, use [trackNextBackgroundForeground]
     */
    fun startObserving()

    /**
     * Tracks the next background of the application. Automatically tracks
     * the subsequent foreground of the application and stops collection.
     * NOTE: This API only collects one bg -> fg event and is not a continuous listener
     */
    fun trackNextBackgroundForeground()
}

/**
 * An application lifecycle tracker that tracks the fg / bg of the application
 * and records it for analytics.
 *
 */
internal class AnalyticsApplicationLifecycleTrackerImpl(
    private val lifecycle: Lifecycle,
    private val fgBgListener: SingleObserverBackgroundThenForegroundAnalyticsListener,
) : AnalyticsApplicationLifecycleTracker {
    companion object {
        private const val TAG = "AnalyticsApplicationLifecycleTrackerImpl"
    }

    private val scope = CoroutineScope(DispatcherProvider().mainImmediate)
    private var hasRegisteredListener = false

    override fun startObserving() {
        MolocoLogger.debug(TAG, "Start observing application lifecycle events")
        scope.launch {
            registerObserver()
        }
    }

    override fun trackNextBackgroundForeground() {
        scope.launch {
            MolocoLogger.debug(TAG, "Tracking next bg / fg of the application")
            registerObserver()
            fgBgListener.trackNext()
        }
    }

    @MainThread
    private fun registerObserver() {
        if (!hasRegisteredListener) {
            MolocoLogger.debug(TAG, "Observing application lifecycle events")
            lifecycle.addObserver(fgBgListener)
            hasRegisteredListener = true
        }
    }
}

/**
 * A custom single-observer background / foreground listener that records the background event before foreground
 * event into the analytics service. It stops observing after a single background -> foreground
 * is recorded
 */
private const val TAG = "SingleObserverBackgroundThenForegroundAnalyticsListener"
internal class SingleObserverBackgroundThenForegroundAnalyticsListener(
    private val analyticsService: AnalyticsService,
    private val timeProviderService: TimeProviderService,
) : DefaultLifecycleObserver {

    private var lastBgTimestamp: Long? = null
    private var trackNext = false

    internal fun trackNext() {
        trackNext = true
    }

    override fun onStart(owner: LifecycleOwner) {
        super.onStart(owner)
        MolocoLogger.debug(TAG, "Application onStart")
        // We specially ensure that application is backgrounded before foreground because
        // when we register the observer the application is already in foreground. So we want
        // to observe for how long was the user outside of the app
        val lastBgTimestampEvent = lastBgTimestamp
        if (lastBgTimestampEvent != null) {
            MolocoLogger.debug(TAG, "Background event has been recorded, recording foreground")
            val eventTimestamp = timeProviderService()
            // Start tracking the application state only after the click background
            analyticsService.recordApplicationForeground(eventTimestamp, lastBgTimestampEvent)
            lastBgTimestamp = null
            trackNext = false
        }
    }

    override fun onStop(owner: LifecycleOwner) {
        super.onStop(owner)
        MolocoLogger.debug(TAG, "Application onStop")
        if (trackNext) {
            MolocoLogger.debug(TAG, "Tracking of event is true. Recording background")
            val eventTimestamp = timeProviderService()
            lastBgTimestamp = eventTimestamp
            analyticsService.recordApplicationBackground(eventTimestamp)
        }
    }
}
