package com.moloco.sdk.internal.services.bidtoken

import com.google.protobuf.InvalidProtocolBufferException
import com.moloco.sdk.BidToken
import com.moloco.sdk.acm.CountEvent
import com.moloco.sdk.internal.Error
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.Result
import com.moloco.sdk.internal.appendXMolocoUserAgent
import com.moloco.sdk.internal.client_metrics_data.AcmCount
import com.moloco.sdk.internal.client_metrics_data.AcmResultTag
import com.moloco.sdk.internal.client_metrics_data.AcmTag
import com.moloco.sdk.internal.client_metrics_data.AcmTimer
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.publisher.MolocoAdError
import com.moloco.sdk.xenoss.sdkdevkit.android.core.requestTimeoutMillis
import io.ktor.client.HttpClient
import io.ktor.client.call.body
import io.ktor.client.plugins.HttpRequestTimeoutException
import io.ktor.client.request.headers
import io.ktor.client.request.post
import io.ktor.client.request.setBody
import io.ktor.http.ContentType
import io.ktor.http.HttpStatusCode
import io.ktor.http.contentType
import kotlinx.coroutines.delay
import kotlinx.coroutines.withContext
import java.net.UnknownHostException
import com.moloco.sdk.acm.AndroidClientMetrics as acm

/**
 * A stateless API client for fetching the bid token from the Moloco Cloud
 * and parsing the response.
 */
private const val BID_TOKEN_URL = "https://sdkapi.dsp-api.moloco.com/v3/bidtoken"

internal enum class BidTokenApiErrorType(val errorCode: Int) {
    UNKNOWN(-100),
    HTTP_REQUEST_TIMEOUT(-101),
    UNKNOWN_HOST(-102),
}

internal interface BidTokenApi {
    suspend fun fetchBidToken(): Result<BidTokenResponseComponents, Error>
}

internal data class BidTokenHttpRequestInfo(
    val requestTimeoutMillis: Long,
    val fetchRetryMax: Int,
    val fetchRetryDelayMillis: Long,
)

internal data class BidTokenDeviceRequestInfo(
    val language: String,
    val osVersion: String,
    val make: String,
    val model: String,
    val hardwareVersion: String,
)

internal class BidTokenApiImpl(
    private val sdkVersion: String,
    private val httpClient: HttpClient,
    private val httpRequestInfo: BidTokenHttpRequestInfo,
    private val deviceRequestInfo: BidTokenDeviceRequestInfo,
) : BidTokenApi {
    private val TAG = "BidTokenApi"

    /**
     * Retry logic JIRA [task](https://mlc.atlassian.net/browse/SDK-1446)
     */
    override suspend fun fetchBidToken(): Result<BidTokenResponseComponents, Error> = withContext(DispatcherProvider().io) {
        var lastResult: Result<BidTokenResponseComponents, Error> = Result.Failure(
            Error("retry max parameter is 0", MolocoAdError.ErrorType.UNKNOWN.errorCode)
        )

        repeat(httpRequestInfo.fetchRetryMax) {attempt ->
            MolocoLogger.info(TAG, "Fetching bidtoken, attempt #$attempt")
            val serverBidTokenApiTimer = acm.startTimerEvent(AcmTimer.ServerBidTokenApiFetchTime.eventName)
            val result = fetchBidTokenWork(BID_TOKEN_URL)
            lastResult = result

            MolocoLogger.info(TAG, "Received bidtoken fetch result: $result")

            when (result) {
                // Do not retry for 4xx http status codes - return immediately.
                is Result.Failure -> {
                    val failureCode = result.value.failureCode
                    acm.recordTimerEvent(serverBidTokenApiTimer
                        .withTag(AcmTag.Result.tagName, AcmResultTag.failure.name)
                        .withTag(AcmTag.Reason.tagName, failureCode.toString())
                        .withTag(AcmTag.RetryAttempt.tagName, attempt.toString())
                    )
                    acm.recordCountEvent(CountEvent(AcmCount.ServerBidTokenApiFetch.eventName)
                        .withTag(AcmTag.Result.tagName, AcmResultTag.failure.name)
                        .withTag(AcmTag.Reason.tagName, failureCode.toString())
                        .withTag(AcmTag.RetryAttempt.tagName, attempt.toString())
                    )

                    if (failureCode in 400..499 && failureCode != 429) {
                        MolocoLogger.info(TAG, "Received 4xx error: $failureCode")
                        return@withContext result
                    } else {
                        MolocoLogger.info(TAG, "Received non-4xx or $failureCode error: $failureCode")
                    }
                }

                // Return with success immediately.
                is Result.Success -> {
                    acm.recordTimerEvent(serverBidTokenApiTimer
                        .withTag(AcmTag.Result.tagName, AcmResultTag.success.name)
                        .withTag(AcmTag.RetryAttempt.tagName, attempt.toString())
                    )
                    acm.recordCountEvent(CountEvent(AcmCount.ServerBidTokenApiFetch.eventName)
                        .withTag(AcmTag.Result.tagName, AcmResultTag.success.name)
                        .withTag(AcmTag.RetryAttempt.tagName, attempt.toString())
                    )
                    return@withContext result
                }
            }

            // Otherwise retry for non 4xx errors.

            // Linear backoff for retry delay.
            val delay = httpRequestInfo.fetchRetryDelayMillis * (attempt + 1)
            MolocoLogger.info(TAG, "Retrying after delay: $delay")
            delay(delay)
        }

        lastResult
    }

    private suspend fun fetchBidTokenWork(
        bidTokenUrl: String,
    ): Result<BidTokenResponseComponents, Error> {
        val response = try {
            httpClient.post(bidTokenUrl) {
                headers {
                    appendXMolocoUserAgent(sdkVersion, deviceRequestInfo.osVersion)
                }

                contentType(ContentType.Application.ProtoBuf)
                setBody(bidTokenRequestBody())
                requestTimeoutMillis(httpRequestInfo.requestTimeoutMillis)
            }
        } catch (e: HttpRequestTimeoutException) {
            MolocoLogger.error(TAG, "Request timeout exception", e)
            return Result.Failure(Error("bidtoken request failed due to timeout", BidTokenApiErrorType.HTTP_REQUEST_TIMEOUT.errorCode))
        } catch(e: UnknownHostException) {
            MolocoLogger.error(TAG, "Unknown Host Request exception", e)
            return Result.Failure(Error("bidtoken request failed due to not being able to connect to host", BidTokenApiErrorType.UNKNOWN_HOST.errorCode))
        } catch (e: Exception) {
            MolocoLogger.error(TAG, "Bid Token API Request exception", e)
            return Result.Failure(Error("bidtoken request failed due to unknown exception", BidTokenApiErrorType.UNKNOWN.errorCode))
        }

        return if (response.status == HttpStatusCode.OK) {
            try {
                val bidResponse = BidToken.BidTokenResponseV3.parseFrom(response.body<ByteArray>())
                Result.Success(BidTokenResponseComponents(bidResponse.bidToken, bidResponse.pk, bidResponse.bidTokenConfig()))
            } catch (e: InvalidProtocolBufferException) {
                /* Protobuff could not parse the response body.
                 * Failure code 400 is used, in order for this request not to be repeated in upstream code.
                 */
                Result.Failure(Error("Bidtoken parsing failed. Reason: $e", HttpStatusCode.BadRequest.value))
            }
        } else {
            Result.Failure(Error("bidtoken request failed", response.status.value))
        }
    }

    private fun bidTokenRequestBody(): ByteArray =
        BidToken.BidTokenRequestV3.newBuilder().build().toByteArray()
}
