package com.moloco.sdk.internal.services.bidtoken

import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.publisher.InitializationHandler
import com.moloco.sdk.internal.services.TimeProviderService
import com.moloco.sdk.publisher.Initialization
import com.moloco.sdk.publisher.Moloco
import com.moloco.sdk.publisher.MolocoAdError
import com.moloco.sdk.publisher.MolocoBidTokenListener

internal interface BidTokenHandler {
    /**
     * Fetches the bid token and returns it to the listener
     */
    suspend fun handleBidTokenRequest(listener: MolocoBidTokenListener)
}

internal class BidTokenHandlerImpl(
    private val bidTokenService: BidTokenService,
    private val initializationHandler: InitializationHandler,
    private val timeProviderService: TimeProviderService
) : BidTokenHandler {
    private val TAG = "BidTokenHandlerImpl"

    override suspend fun handleBidTokenRequest(listener: MolocoBidTokenListener) {
        // If the SDK cannot initialize at all, don't allow making bid request that will eventually fail
        if (initializationHandler.canInitialize().not()) {
            val error = MolocoAdError.ErrorType.SDK_PERSISTENT_HTTP_REQUEST_FAILED_TO_INIT
            MolocoLogger.info(TAG, "Bid token cannot be fetched because SDK initialization cannot happen due to WM issue")
            listener.onBidTokenResult("", error)
            return
        }

        // If SDK initialization has failed, don't allow making bid request by sending failed bid token
        // MAX does not check for Moloco SDK initialization status and keeps sending us the sdk ad load request
        if (initializationHandler.initializationState.value == Initialization.FAILURE) {
            MolocoLogger.info(TAG, "Bid token cannot be fetched because SDK initialization has failed")
            listener.onBidTokenResult("", MolocoAdError.ErrorType.SDK_INIT_ERROR)
            return
        }

        val bidTokenLoadStart = timeProviderService.currentTime()
        val bidToken = bidTokenService.bidToken()
        val bidTokenLoadEnd = timeProviderService.currentTime() - bidTokenLoadStart

        MolocoLogger.info(TAG, "Bid token fetched in $bidTokenLoadEnd ms")

        val error = if (bidToken.isEmpty()) MolocoAdError.ErrorType.AD_SIGNAL_COLLECTION_FAILED else null
        MolocoLogger.info(TAG, "Returning bid token result, hasError: ${error != null}, SDK init complete: ${Moloco.isInitialized}")
        listener.onBidTokenResult(bidToken, error)
    }
}