package com.moloco.sdk.internal.services.bidtoken

import androidx.annotation.VisibleForTesting
import com.moloco.sdk.BuildConfig
import com.moloco.sdk.acm.AndroidClientMetrics
import com.moloco.sdk.acm.CountEvent
import com.moloco.sdk.acm.TimerEvent
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.client_metrics_data.AcmCount
import com.moloco.sdk.internal.client_metrics_data.AcmResultTag
import com.moloco.sdk.internal.client_metrics_data.AcmTag
import com.moloco.sdk.internal.client_metrics_data.AcmTimer
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import com.moloco.sdk.acm.AndroidClientMetrics as acm

// The implementation cannot be provided by DI because this service can be
// called even before SDK is initialized (which is what initializes DI).
internal fun BidTokenService(): BidTokenService = Instance

/**
 * A stateful service that manages the BidToken at the SDK layer. The BidToken
 * can expire based on numerous factors and this service is responsible for
 * refreshing the token when necessary.
 */
internal interface BidTokenService {
    /**
     * Returns the bid token that should be used for BidRequest
     */
    suspend fun bidToken(): String
}

private val Instance by lazy {
    MolocoLogger.info("BidTokenService", "Creating BidTokenService instance")
    BidTokenServiceImpl(
        serverBidTokenService = ServerBidTokenService.create(),
        clientBidTokenService = ClientBidTokenService.create(),
    )
}

@VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
internal class BidTokenServiceImpl(
    private val serverBidTokenService: ServerBidTokenService,
    private val clientBidTokenService: ClientBidTokenService,
) : BidTokenService {
    private val TAG = "BidTokenServiceImpl"

    private val mutex = Mutex()

    override suspend fun bidToken(): String {
        // For thread safety
        mutex.withLock {
            val bidTokenTimer = AndroidClientMetrics.startTimerEvent(AcmTimer.BidTokenFetch.eventName)
            val serverBidTokenResponse: BidTokenFetchResult = serverBidTokenService.bidToken()
            val serverBidToken = serverBidTokenResponse.bidToken
            var failureReason:String? = null

            val bidToken = if (serverBidToken.isNotEmpty()) {
                val rsaPublicKey = serverBidTokenResponse.publicKey
                val bidTokenConfig = serverBidTokenResponse.bidTokenConfig
                val clientBidToken = clientBidTokenService.bidToken(rsaPublicKey, bidTokenConfig).getOrDefault("")

                if (clientBidToken.isEmpty()) {
                    MolocoLogger.error(TAG, "CBT has error")
                    failureReason = "client"
                    ""
                } else {
                    "$serverBidToken:$clientBidToken"
                }
            } else {
                failureReason = "server"
                ""
            }
            if (BuildConfig.DEBUG) {
                // Don't want to log our bid token in release variant
                MolocoLogger.debug(TAG, "Proposed Bid Token: $bidToken")
            }

            emitBidTokenFetchAnalytics(bidTokenTimer, failureReason)

            return bidToken
        }
    }

    private fun emitBidTokenFetchAnalytics(bidTokenTimer: TimerEvent, failureReason: String?) {
        // Mediation is part of the ACM library and will be passed along with all events
        if (failureReason != null) {
            // Record the failed
            acm.recordCountEvent(
                CountEvent(AcmCount.BidTokenFetch.eventName)
                    .withTag(AcmTag.Result.tagName, AcmResultTag.failure.name)
                    .withTag(AcmTag.Reason.tagName, failureReason)
            )

            acm.recordTimerEvent(
                bidTokenTimer
                    .withTag(AcmTag.Result.tagName, AcmResultTag.failure.name)
                    .withTag(AcmTag.Reason.tagName, failureReason)
            )
        } else {
            // Record the success event
            acm.recordCountEvent(
                CountEvent(AcmCount.BidTokenFetch.eventName)
                    .withTag(AcmTag.Result.tagName, AcmResultTag.success.name)
            )

            acm.recordTimerEvent(bidTokenTimer
                .withTag(AcmTag.Result.tagName, AcmResultTag.success.name))
        }
    }
}
