package com.moloco.sdk.internal.services.bidtoken

import android.os.BatteryManager
import com.google.protobuf.ByteString
import com.moloco.sdk.BidToken
import com.moloco.sdk.internal.services.AdData
import com.moloco.sdk.internal.services.DeviceInfo
import com.moloco.sdk.internal.services.DeviceInfoService
import com.moloco.sdk.internal.services.MuteSwitchState
import com.moloco.sdk.internal.services.NetworkInfo
import com.moloco.sdk.internal.services.ScreenInfo
import com.moloco.sdk.internal.services.ScreenInfoService
import com.moloco.sdk.internal.services.ScreenOrientation
import com.moloco.sdk.internal.services.bidtoken.providers.ClientSignals
import com.moloco.sdk.service_locator.SdkObjectFactory
import java.util.Date
import java.util.TimeZone

/**
 * Client side token builder as per the proposal here - https://docs.google.com/document/d/1elJPg3JNxJ4s6qo1OdHDlo6mOgTNtBOHwctkJDl5r4o/edit#heading=h.rvg2y6iz8zo9
 */
internal interface ClientBidTokenBuilder {
    companion object {
        fun create(): ClientBidTokenBuilder {
            return ClientBidTokenBuilderImpl(
                SdkObjectFactory.DeviceAndApplicationInfo.deviceInfoSingleton,
                SdkObjectFactory.DeviceAndApplicationInfo.screenInfoSingleton)
        }
    }
    fun buildBidToken(bidTokenComponents: ByteArray, secret: ByteArray): ByteArray
    fun buildClientBidTokenComponent(clientSignals: ClientSignals, bidTokenConfig: BidTokenConfig): BidToken.ClientBidTokenComponents

    fun parseClientBidTokenComponent(payload: ByteArray): BidToken.ClientBidTokenComponents
    fun parse(payload: ByteArray): BidToken.ClientBidToken
}

internal class ClientBidTokenBuilderImpl(
    private val deviceInfoService: DeviceInfoService,
    private val screenInfoService: ScreenInfoService,
): ClientBidTokenBuilder {
    override fun buildBidToken(
        bidTokenComponents: ByteArray,
        secret: ByteArray,
    ): ByteArray {
        return BidToken.ClientBidToken.newBuilder().apply {
            es = ByteString.copyFrom(secret)
            payload = ByteString.copyFrom(bidTokenComponents)
        }.build().toByteArray()
    }

    override fun buildClientBidTokenComponent(
        clientSignals: ClientSignals,
        bidTokenConfig: BidTokenConfig
    ): BidToken.ClientBidTokenComponents {
        val deviceInfo = deviceInfoService.deviceInfo()
        val screenInfo = screenInfoService.screenInfo()
        return BidToken.ClientBidTokenComponents.newBuilder().apply {
            info = BidToken.ClientBidTokenComponents.SdkInfo.newBuilder().apply {
                initialized = clientSignals.sdkInitialized
            }.build()

            memoryInfo = BidToken.ClientBidTokenComponents.MemoryInfo.newBuilder().apply {
                lowMem = clientSignals.memoryInfo.lowMemory
                lowMemThresholdBytes = clientSignals.memoryInfo.threshold
                totalMemBytes = clientSignals.memoryInfo.totalMem
            }.build()

            dirInfo = BidToken.ClientBidTokenComponents.DirInfo.newBuilder().apply {
                dsizeBytes = clientSignals.appDirInfo.appDirSize
            }.build()

            networkInfo = BidToken.ClientBidTokenComponents.NetworkInfo.newBuilder().apply {
                mcc = clientSignals.networkInfoSignal.mobileCountryCode ?: -1
                mnc = clientSignals.networkInfoSignal.mobileNetworkCode ?: -1
                restricted = clientSignals.networkInfoSignal.networkRestricted
                type = when (clientSignals.networkInfoSignal.networkType) {
                    is NetworkInfo.CellularNetwork -> BidToken.ClientBidTokenComponents.NetworkInfo.ConnectionType.CELLULAR
                    NetworkInfo.NoNetwork -> BidToken.ClientBidTokenComponents.NetworkInfo.ConnectionType.NO_NETWORK
                    NetworkInfo.WifiNetwork -> BidToken.ClientBidTokenComponents.NetworkInfo.ConnectionType.WIFI
                }
            }.build()

            batteryInfo = BidToken.ClientBidTokenComponents.BatteryInfo.newBuilder().apply {
                maxBatteryLevel = clientSignals.batteryInfoSignal.maxBatteryLevel
                batteryStatus = when (clientSignals.batteryInfoSignal.batteryStatus) {
                    BatteryManager.BATTERY_STATUS_CHARGING -> BidToken.ClientBidTokenComponents.BatteryInfo.BatteryStatus.CHARGING
                    BatteryManager.BATTERY_STATUS_DISCHARGING -> BidToken.ClientBidTokenComponents.BatteryInfo.BatteryStatus.DISCHARGING
                    BatteryManager.BATTERY_STATUS_NOT_CHARGING -> BidToken.ClientBidTokenComponents.BatteryInfo.BatteryStatus.NOT_CHARGING
                    BatteryManager.BATTERY_STATUS_FULL -> BidToken.ClientBidTokenComponents.BatteryInfo.BatteryStatus.FULL
                    else -> BidToken.ClientBidTokenComponents.BatteryInfo.BatteryStatus.UNKNOWN
                }
                lowPowMode = clientSignals.batteryInfoSignal.isPowerSaveMode
            }.build()

            adInfo = BidToken.ClientBidTokenComponents.AdvertisingInfo.newBuilder().apply {
                when (val adData = clientSignals.adDataSignal) {
                    is AdData.Available -> {
                        dnt = false
                        id = adData.id
                    }
                    AdData.Unavailable -> {
                        dnt = true
                    }
                }
            }.build()

            privacy = BidToken.ClientBidTokenComponents.Privacy.newBuilder().apply {
                clientSignals.privacySettings.isAgeRestrictedUser?.let {
                    coppa = it
                }
                clientSignals.privacySettings.isUserConsent?.let {
                    gdpr = it
                }
                clientSignals.privacySettings.isDoNotSell?.let {
                    ccpa = it
                }
                clientSignals.privacySettings.TCFConsent?.let {
                    tcfConsentString = it
                }
                usPrivacy = clientSignals.privacySettings.usPrivacy
            }.build()

            device = BidToken.ClientBidTokenComponents.Device.newBuilder().apply {
                // 1
                language = deviceInfo.language
                // 2
                osv = deviceInfo.osVersion
                // 3
                make = deviceInfo.manufacturer
                // 4
                model = deviceInfo.model
                // 5
                hwv = deviceInfo.hwVersion
                // 6
                carrier = deviceInfo.mobileCarrier
                // 7
                devicetype = if (deviceInfo.isTablet) 5 else 1
                // 8
                js = 1
                // 9
                geo = BidToken.ClientBidTokenComponents.Geo.newBuilder().apply {
                    utcoffset = TimeZone.getDefault().getOffset(Date().time) / 60000
                }.build()
                // 10
                w = screenInfo.screenWidthPx
                // 11
                h = screenInfo.screenHeightPx
                // 12
                pxratio = screenInfo.density.toDouble()
                // 13
                ppi = screenInfo.dpi
                // 14
                os = deviceInfo.os
                // 15
                if (bidTokenConfig.dbtEnabled) {
                    dbt = deviceInfo.dbtMs.millisToNanos()
                }
                // 16
                ortn = clientSignals.deviceSignal.orientation.toProtoOrientation()
                // 17
                hasGy = deviceInfoService.hasGyroscope()
                // 18
                kbLoc = clientSignals.deviceSignal.keyboardLocale
                // 19
                locale = clientSignals.deviceSignal.locale
                // 20
                xdpi = screenInfo.xdpi
                // 21
                ydpi = screenInfo.ydpi
            }.build()

            audioInfo = BidToken.ClientBidTokenComponents.AudioInfo.newBuilder().apply {
                muteSwitch = clientSignals.audioSignal.muteSwitchState.toProtoMuteSwitchState()
                vol = clientSignals.audioSignal.mediaVolume
            }.build()

            accessibilityInfo = BidToken.ClientBidTokenComponents.AccessibilityInfo.newBuilder().apply {
                fontScale = clientSignals.accessibilitySignal.fontScale
                accessibilityLargePointerIcon = clientSignals.accessibilitySignal.accessibilityLargePointerIcon
                accessibilityCaptioningEnabled = clientSignals.accessibilitySignal.accessibilityCaptioningEnabled
                reduceBrightColorsActivated = clientSignals.accessibilitySignal.reduceBrightColorsActivated
            }.build()
        }.build()

    }

    override fun parse(payload: ByteArray): BidToken.ClientBidToken {
        return BidToken.ClientBidToken.parseFrom(payload)
    }

    override fun parseClientBidTokenComponent(payload: ByteArray): BidToken.ClientBidTokenComponents {
        return BidToken.ClientBidTokenComponents.parseFrom(payload)
    }

    private fun Long.millisToNanos(): Long = this * 1_000_000

    private fun ScreenOrientation.toProtoOrientation(): BidToken.ClientBidTokenComponents.Device.Orientation =
        when (this) {
            ScreenOrientation.UNKNOWN -> BidToken.ClientBidTokenComponents.Device.Orientation.UNKNOWN
            ScreenOrientation.PORTRAIT -> BidToken.ClientBidTokenComponents.Device.Orientation.PORTRAIT
            ScreenOrientation.LANDSCAPE -> BidToken.ClientBidTokenComponents.Device.Orientation.LANDSCAPE
        }

    private fun MuteSwitchState.toProtoMuteSwitchState(): BidToken.ClientBidTokenComponents.AudioInfo.MuteSwitchState =
        when (this) {
            MuteSwitchState.SILENT -> BidToken.ClientBidTokenComponents.AudioInfo.MuteSwitchState.SILENT
            MuteSwitchState.VIBRATE -> BidToken.ClientBidTokenComponents.AudioInfo.MuteSwitchState.VIBRATE
            MuteSwitchState.NORMAL -> BidToken.ClientBidTokenComponents.AudioInfo.MuteSwitchState.NORMAL
        }
}