package com.moloco.sdk.internal.services.encryption

import android.util.Base64
import androidx.annotation.VisibleForTesting
import com.moloco.sdk.BuildConfig
import java.security.KeyFactory
import java.security.PrivateKey
import java.security.PublicKey
import java.security.spec.PKCS8EncodedKeySpec
import java.security.spec.X509EncodedKeySpec
import javax.crypto.Cipher
import javax.crypto.KeyGenerator
import javax.crypto.spec.IvParameterSpec
import javax.crypto.spec.SecretKeySpec

internal typealias EncryptedAESSecret = ByteArray
internal interface EncryptionService {
    companion object {
        fun create(): EncryptionService = EncryptionServiceImpl()
    }

    fun aesEncrypt(data: ByteArray): ByteArray
    fun aesDecrypt(data: ByteArray): String
    fun aesDecrypt(data: ByteArray, secret: ByteArray): String

    fun rsaEncryptedAesKey(rsaPublicKey: String): EncryptedAESSecret
    fun encryptRSA(input: ByteArray, publicKey: String): ByteArray
    fun decryptRSA(input: ByteArray, privateKey: String): ByteArray

    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    val aesSecret: SecretKeySpec

    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    val aesIv: IvParameterSpec
}

internal class EncryptionServiceImpl : EncryptionService {

    private val RSA by lazy {
        // "RSA"
        Base64.decode(byteArrayOf(85, 108, 78, 66), Base64.DEFAULT).decodeToString()
    }

    private val RSA_CIPHER by lazy {
        // RSA/ECB/OAEPWithSHA-256AndMGF1Padding
        // Not base64 encoded
        byteArrayOf(82, 83, 65, 47, 69, 67, 66, 47, 79, 65, 69, 80, 87, 105, 116, 104, 83, 72, 65, 45, 50, 53, 54, 65, 110, 100, 77, 71, 70, 49, 80, 97, 100, 100, 105, 110, 103).decodeToString()
    }

    private val AES by lazy {
    //    "AES"
        byteArrayOf(65, 69, 83).decodeToString()
    }

    private val AES_CIPHER by lazy {
        // "AES/GCM/NoPadding"
        Base64.decode(byteArrayOf(81, 85, 86, 84, 76, 48, 100, 68, 84, 83, 57, 79, 98, 49, 66, 104, 90, 71, 82, 112, 98, 109, 99, 61), Base64.DEFAULT).decodeToString()
    }

    override val aesSecret: SecretKeySpec = generateSecretKeyWithRandomAES()
    override val aesIv: IvParameterSpec
        by lazy {
            val salt = Base64.decode(BuildConfig.MOLOCO_SDK_BIDTOKEN_SALT, Base64.DEFAULT)
            IvParameterSpec(salt)
        }

    override fun aesEncrypt(data: ByteArray): ByteArray {
        val cipher = Cipher.getInstance(AES_CIPHER)
        cipher.init(Cipher.ENCRYPT_MODE, aesSecret, aesIv)
        return cipher.doFinal(data)
    }

    override fun aesDecrypt(data: ByteArray): String {
        val cipher = Cipher.getInstance(AES_CIPHER)
        cipher.init(Cipher.DECRYPT_MODE, aesSecret, aesIv)
        return cipher.doFinal(data).decodeToString()
    }

    override fun aesDecrypt(data: ByteArray, secret: ByteArray): String {
        val cipher = Cipher.getInstance(AES_CIPHER)
        val aesSecret = SecretKeySpec(secret, AES)
        cipher.init(Cipher.DECRYPT_MODE, aesSecret, aesIv)
        return cipher.doFinal(data).decodeToString()
    }

    override fun rsaEncryptedAesKey(rsaPublicKey: String): EncryptedAESSecret {
        return encryptRSA(aesSecret.encoded, rsaPublicKey)
    }

    override fun encryptRSA(input: ByteArray, publicKey: String): ByteArray {
        val cipher = Cipher.getInstance(RSA_CIPHER)
        cipher.init(Cipher.ENCRYPT_MODE, createRSAPublicKeyFromString(publicKey))
        return cipher.doFinal(aesSecret.encoded)
    }

    override fun decryptRSA(input: ByteArray, privateKey: String): ByteArray {
        val cipher = Cipher.getInstance(RSA_CIPHER)
        cipher.init(Cipher.DECRYPT_MODE, createRSAPrivateKey(privateKey))
        return cipher.doFinal(input)
    }

    private fun generateSecretKeyWithRandomAES(): SecretKeySpec {
        // Generate a random AES secret
        val keyGen = KeyGenerator.getInstance(AES)
        keyGen.init(256)
        val randomAesSecret = keyGen.generateKey().encoded
        return SecretKeySpec(randomAesSecret, AES)
    }

    private fun createRSAPublicKeyFromString(encodedKey: String): PublicKey {
        // Decode the Base64 encoded public key string
        val decodedKey = Base64.decode(encodedKey, Base64.DEFAULT)

        // Generate the PublicKey from the decoded byte array
        val keySpec = X509EncodedKeySpec(decodedKey)
        val keyFactory = KeyFactory.getInstance(RSA)

        return keyFactory.generatePublic(keySpec)
    }

    private fun createRSAPrivateKey(privateKey: String): PrivateKey {
        // Decode the Base64 encoded private key string
        val decodedKey = Base64.decode(privateKey, Base64.DEFAULT)

        // Generate the PrivateKey from the decoded byte array
        val keySpec = PKCS8EncodedKeySpec(decodedKey)
        val keyFactory = KeyFactory.getInstance(RSA)
        return keyFactory.generatePrivate(keySpec)
    }
}