package com.moloco.sdk.internal.services.init

import android.net.Uri
import com.moloco.sdk.MetricsRequest
import com.moloco.sdk.MetricsRequest.SDKInitFailureTrackingRequest
import com.moloco.sdk.MetricsRequest.SDKInitTrackingRequest
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.xenoss.sdkdevkit.android.persistenttransport.HttpRequestClient
import io.ktor.http.ContentType

internal interface InitTrackingApi {
    /**
     * @param region region of notifnt server (US, EU, Asia) based on initialization country. "US" by default if region info unavailable.
     */
    suspend fun notifySuccess(latency: Long)

    /**
     * @param failureType reason/type for failed initialization.
     */
    suspend fun notifyFailure(failureType: InitFailure, latency: Long)
}

private const val TAG = "InitTrackingApi"
internal class InitTrackingApiImpl(
    private val endpoint: String,
    private val httpRequestClient: HttpRequestClient,
) : InitTrackingApi {

    override suspend fun notifySuccess(latency: Long) {
        try {
            MolocoLogger.debug(TAG, "Reporting InitTracking success")

            if (endpoint.isEmpty()) {
                MolocoLogger.debug(TAG, "SDK InitTracking disabled")
                return
            }

            val preparedUrl = Uri.parse(endpoint)
                .buildUpon()
                .build()

            httpRequestClient.sendPost(preparedUrl.toString(), buildSuccessPayload(latency = latency), ContentType.Application.ProtoBuf)
        } catch (e: Exception) {
            MolocoLogger.error(TAG, "Failed to send notifySuccess post request", e)
        }
    }

    override suspend fun notifyFailure(failureType: InitFailure, latency: Long) {
        try {
            when(failureType) {
                is InitFailure.ClientError -> MolocoLogger.debug(TAG, "Reporting InitTracking client failure: ${failureType.type}")
                is InitFailure.ServerError -> MolocoLogger.debug(TAG, "Reporting InitTracking server failure: ${failureType.statusCode}")
            }

            if (endpoint.isEmpty()) {
                MolocoLogger.debug(TAG, "SDK InitTracking disabled")
                return
            }

            val preparedUrl = Uri.parse(endpoint)
                .buildUpon()
                .build()

            val body = buildFailurePayload(failureType = failureType, latency = latency)
            httpRequestClient.sendPost(preparedUrl.toString(), body, ContentType.Application.ProtoBuf)
        } catch (e: Exception) {
            MolocoLogger.error(TAG, "Failed to send notifyFailure post request", e)
        }
    }

    private fun buildSuccessPayload(latency: Long): ByteArray {
        return SDKInitTrackingRequest.newBuilder().apply {
            this.latencyMs = latency
            success = MetricsRequest.SDKInitSuccessTrackingRequest.newBuilder().build()
        }.build().toByteArray()
    }

    private fun buildFailurePayload(failureType: InitFailure, latency: Long): ByteArray {
        return SDKInitTrackingRequest.newBuilder().apply {
            failure = SDKInitFailureTrackingRequest.newBuilder().apply {
                when(failureType) {
                    is InitFailure.ClientError -> this.clientError = SDKInitFailureTrackingRequest.ClientError.newBuilder().apply {
                        clientFailureType = failureType.type.toProtoFailureType()
                    }.build()
                    is InitFailure.ServerError -> this.serverError = SDKInitFailureTrackingRequest.ServerError.newBuilder().apply {
                        serverHttpStatus = failureType.statusCode
                    }.build()
                }
            }.build()
            this.latencyMs = latency
        }.build().toByteArray()
    }

    private fun ClientFailureType.toProtoFailureType(): SDKInitFailureTrackingRequest.ClientError.ClientErrorTypes = when (this) {
        ClientFailureType.Unknown -> SDKInitFailureTrackingRequest.ClientError.ClientErrorTypes.UNKNOWN
        ClientFailureType.RequestTimeout -> SDKInitFailureTrackingRequest.ClientError.ClientErrorTypes.HTTP_REQUEST_TIMEOUT
        ClientFailureType.UnknownHostHttpError -> SDKInitFailureTrackingRequest.ClientError.ClientErrorTypes.HTTP_UKNOWN_HOST
        ClientFailureType.HttpSocketError -> SDKInitFailureTrackingRequest.ClientError.ClientErrorTypes.HTTP_SOCKET
        ClientFailureType.HttpSslError -> SDKInitFailureTrackingRequest.ClientError.ClientErrorTypes.HTTP_SSL_ERROR
        ClientFailureType.PersistentHttpUnavailableError -> SDKInitFailureTrackingRequest.ClientError.ClientErrorTypes.ANDROID_WORK_MANAGER_ISSUE
        // no else branch, explicitly handle all mapping cases here
    }
}
