package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer

import android.content.Context
import android.view.View
import android.view.ViewGroup
import android.view.ViewGroup.LayoutParams.MATCH_PARENT
import android.webkit.WebView
import android.widget.FrameLayout
import androidx.annotation.VisibleForTesting
import androidx.compose.ui.platform.ComposeView
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.cancel
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.launch
import kotlin.time.Duration

/**
 * Banner container for ad rendering. Works almost the same as [FullscreenAd] implementations <br>
 * The only difference is that instead of calling show() function, call [android.view.View.setVisibility] = true (false or detach from view hierarchy stops displaying an ad)
 */
internal abstract class XenossBannerView<T : AdShowListener>(
    context: Context,
    private val scope: CoroutineScope,
) : FrameLayout(context), CreativeTypeProvider, AdLoad, AdDisplayState, Destroyable {

    open var adShowListener: T? = null


    @VisibleForTesting(otherwise = VisibleForTesting.PROTECTED)
    var adView: View? = null
        set(adView) {
            val oldAdView = field
            field = adView

            removeAllViews()
            // TODO. Just to be on the safe side. Should be called outside though
            //  and in the perfect world, we should not know anything about implementation details (ComposeView).
            // https://developer.android.com/reference/kotlin/androidx/compose/ui/platform/ComposeView:
            // Call disposeComposition to dispose of the underlying composition earlier,
            // or if the view is never initially attached to a window. (The requirement to dispose of the composition explicitly in the event that the view is never (re)attached is temporary.)
            (oldAdView as? ComposeView)?.disposeComposition()

            adView?.let {
                addView(it, ViewGroup.LayoutParams(MATCH_PARENT, MATCH_PARENT))
            }
        }

    protected abstract val adLoader: AdLoad

    override val isLoaded by lazy { adLoader.isLoaded }

    override fun load(timeout: Duration, listener: AdLoad.Listener?) {
        scope.launch {
            adLoader.load(timeout, listener)
            isLoaded.first { it }
            prepareAdViewForDisplay()
        }
    }

    abstract fun prepareAdViewForDisplay()

    override fun onVisibilityChanged(changedView: View, visibility: Int) {
        super.onVisibilityChanged(changedView, visibility)
        isViewVisible.value = visibility == VISIBLE
    }

    /**
     * Denotes if the view is visible.
     * Note: If its parent is [View.INVISIBLE] or [View.GONE], this will still return `true` if its visibility is set to [View.VISIBLE]
     */
    private val isViewVisible = MutableStateFlow(false)

    /**
     * Denotes if the view is visible.
     * Note: If its parent is [View.INVISIBLE] or [View.GONE], this will still return `true` if its visibility is set to [View.VISIBLE]
     */
    override val isAdDisplaying by lazy {
        isLoaded.combine(isViewVisible) { isLoaded, isViewShown ->
            isLoaded && isViewShown
        }.stateIn(
            scope,
            SharingStarted.Eagerly,
            false
        )
    }

    /**
     * Adds the `WebView` as a child of a container wrapper (`FrameLayout`) to address layout issues.
     *
     * **Background:**
     * Adding a `WebView` directly without a wrapper can result in `window.innerHeight` being
     * temporarily set to `0` during HTML content layout passes. This behavior can cause
     * unexpected bugs in JavaScript-based creatives, particularly those relying on `window.innerHeight`
     * for layout or rendering logic.
     *
     * **Real-World Example:**
     * - In the "King Dungeon" MRAID game, when MRAID visibility is set to `true`, the
     *   JavaScript WebGL framebuffer code breaks due to transient layout passes where
     *   `window.innerHeight == 0`.
     *
     * **Solution:**
     * Wrapping the `WebView` in a container like `FrameLayout` ensures that
     * `window.innerHeight` values remain stable during layout passes, avoiding
     * transient `0` values and associated rendering issues.
     */
    protected fun wrapWebView(context:Context, webView: WebView): FrameLayout {
        with(webView) {
            setBackgroundColor(android.graphics.Color.TRANSPARENT)
            visibility = View.VISIBLE
        }

        return FrameLayout(context).apply {
            addView(
                webView,
                ViewGroup.LayoutParams(
                    MATCH_PARENT,
                    MATCH_PARENT
                )
            )
        }
    }

    /**
     * Releases all unmanaged resources and removes itself from the view hierarchy.
     */
    override fun destroy() {
        scope.cancel()
        adView = null

        (parent as? ViewGroup)?.removeView(this)
    }
}
