package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.media

import android.content.Context
import com.moloco.sdk.internal.Error
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.Result
import java.io.File
import java.io.IOException

internal const val FILE_CREATE_SECURITY_ERROR = 100
internal const val FILE_CREATE_STORAGE_DIRECTORY_IO_ERROR = 101
internal const val FILE_CREATE_STORAGE_DIRECTORY_NOT_AVAILABLE_ERROR = 102
internal const val FILE_CREATE_STORAGE_DIRECTORY_UNKNOWN_ERROR = 200

/**
 * Factory method that creates a [MediaCacheLocationProvider] instance
 */
internal fun createMediaCacheLocationProvider(appContext: Context, cacheDir: String): MediaCacheLocationProvider {
    return MediaCacheLocationProviderImpl(appContext, cacheDir)
}

internal interface MediaCacheLocationProvider {
    /**
     * Provides the external cache directory (and creates one if doesn't exist)
     * or return an error with specific code to indicate the failure
     */
    fun externalCacheDirForStorage(): Result<File, Error>

    /**
     * Provides the external cache directory (and creates one if doesn't exist)
     * or return an error with specific code to indicate the failure
     */
    fun internalCacheDirForStorage(): Result<File, Error>
}

/**
 * Implementation of [MediaCacheLocationProvider] that provides the external cache directory
 * or return an error with specific code to indicate the failure
 */
internal class MediaCacheLocationProviderImpl(val appContext: Context, val cacheDir: String) :
    MediaCacheLocationProvider {
    private val TAG = "MediaCacheLocationProviderImpl"

    override fun externalCacheDirForStorage(): Result<File, Error> {
        // Try to create a file in the external cache directory
        try {
            val externalCacheDir = appContext.externalCacheDir
            if (externalCacheDir != null) {
                val externalCacheFile = File(
                    externalCacheDir,
                    cacheDir
                )
                externalCacheFile.mkdir()
                if (externalCacheFile.exists()) {
                    return Result.Success(externalCacheFile)
                }
            }

            MolocoLogger.warn(TAG, "Failed to create cache directory in external storage")
            return Result.Failure(
                Error(
                    "Failed to create cache directory in external storage",
                    FILE_CREATE_STORAGE_DIRECTORY_NOT_AVAILABLE_ERROR
                )
            )
        } catch (ioe: IOException) {
            MolocoLogger.error(TAG, "Failed to create cache directory in external storage", ioe)
            return Result.Failure(
                Error(
                    "Failed to create cache directory in external storage",
                    FILE_CREATE_STORAGE_DIRECTORY_IO_ERROR
                )
            )
        } catch (se: SecurityException) {
            MolocoLogger.error(TAG, "Failed to create cache directory in external storage", se)
            return Result.Failure(
                Error(
                    "Failed to create cache directory in external storage",
                    FILE_CREATE_SECURITY_ERROR
                )
            )
        } catch (e: Exception) {
            MolocoLogger.error(TAG, "Failed to create cache directory in external storage", e)
            return Result.Failure(
                Error(
                    "Failed to create cache directory in external storage",
                    FILE_CREATE_STORAGE_DIRECTORY_UNKNOWN_ERROR
                )
            )
        }
    }

    /**
     * Try to create a file in the external cache directory or return an error with specific code to indicate the failure
     */
    override fun internalCacheDirForStorage(): Result<File, Error> {
        // Try to create a file in the external cache directory
        try {
            val internalCacheDir = appContext.cacheDir
            if (internalCacheDir != null) {
                val internalCacheFile = File(appContext.cacheDir, cacheDir)
                internalCacheFile.mkdir()
                if (internalCacheFile.exists()) {
                    MolocoLogger.debug(TAG, "Able to write to internal storage cache directory")
                    return Result.Success(internalCacheFile)
                }
            }

            MolocoLogger.error(TAG, "Failed to create cache directory in internal storage")
            return Result.Failure(
                Error(
                    "Failed to create cache directory in internal storage",
                    FILE_CREATE_STORAGE_DIRECTORY_NOT_AVAILABLE_ERROR
                )
            )
        } catch (ioe: IOException) {
            MolocoLogger.error(TAG, "Failed to create cache directory in external storage", ioe)
            return Result.Failure(
                Error(
                    "Failed to create cache directory in internal storage",
                    FILE_CREATE_STORAGE_DIRECTORY_IO_ERROR
                )
            )
        } catch (se: SecurityException) {
            MolocoLogger.error(TAG, "Failed to create cache directory in external storage", se)
            return Result.Failure(
                Error(
                    "Failed to create cache directory in internal storage",
                    FILE_CREATE_SECURITY_ERROR
                )
            )
        } catch (e: Exception) {
            MolocoLogger.error(TAG, "Failed to create cache directory in external storage", e)
            return Result.Failure(
                Error(
                    "Failed to create cache directory in internal storage",
                    FILE_CREATE_STORAGE_DIRECTORY_UNKNOWN_ERROR
                )
            )
        }
    }
}