package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui

import androidx.compose.foundation.layout.BoxScope
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.runtime.snapshotFlow
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.GoNextAction
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.ad.AdViewModel
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.filter
import kotlinx.coroutines.flow.take

// https://mlc.atlassian.net/browse/SDK-1084
// Main place for skip/close, that is, "next ad part" type buttons of VastRender;
// for linears, companions, DECs etc.
@Composable
internal fun BoxScope.AdGoNextButton(
    adViewModel: AdViewModel,
    currentAdPart: AdViewModel.AdPart?,
    adSkipCountdownButton: AdCountdownButton?,
    adCloseCountdownButton: AdCountdownButton?
) {

    // If its the last ad part - render close
    // if its not the last ad part - render skip
    val countdownButton = when (currentAdPart?.isLastAdPart) {
        true -> adCloseCountdownButton
        false -> adSkipCountdownButton
        null -> null
    }

    // Only enable countdown timer for certain sections
    val enableCountDownTimer = when (currentAdPart) {
        is AdViewModel.AdPart.Companion -> false
        is AdViewModel.AdPart.DEC -> false
        is AdViewModel.AdPart.Linear -> true
        else -> { false }
    }

    countdownButton?.let {
        val goNextButton = @Composable {
            GoNextButton(
                adViewModel.goNextAction,
                adViewModel::goNextAdPartOrDismissAd,
                adViewModel::onButtonRendered,
                countdownButton,
                enableCountDownTimer
            )
        }

        // https://mlc.atlassian.net/browse/SDK-1308
        // Fix for Telepathy tracking not registering new buttons rendering over each new ad part.
        // Similar fix to what we already have in MolocoVastCTA.kt
        // (failed to find JIRA task).
        when (currentAdPart) {
            is AdViewModel.AdPart.Companion -> goNextButton()
            is AdViewModel.AdPart.DEC -> goNextButton()
            is AdViewModel.AdPart.Linear -> goNextButton()
            else -> { /* no-op */ }
        }
    }
}

@Composable
private fun BoxScope.GoNextButton(
    goNextAction: StateFlow<GoNextAction.State>,
    onGoNext: () -> Unit,
    onButtonRendered: (CustomUserEventBuilderService.UserInteraction.Button) -> Unit,
    countdownButton: AdCountdownButton,
    enableCountDownTimer: Boolean
) {
    val goNextActionState by goNextAction.collectAsStateWithLifecycle()

    var countDownTimerStartSeconds by remember { mutableStateOf(0) }

    LaunchedEffect(Unit) {
        // Wait for first non-zero value
        snapshotFlow { (goNextActionState as? GoNextAction.State.Countdown)?.seconds?.toInt() ?: 0 }
            .filter { it > 0 }
            .take(1)
            .collect { value ->
                countDownTimerStartSeconds = value
            }
    }

    val totalCountDownTime = countDownTimerStartSeconds.toUInt()
    val currentCountDownTime = ((goNextActionState as? GoNextAction.State.Countdown)?.seconds?.toInt() ?: 0).toUInt()

    countdownButton(
        // enabled
        goNextActionState is GoNextAction.State.Available,
        // visible
        goNextActionState !is GoNextAction.State.Unavailable,
        // onClick
        onGoNext,
        // onButtonRendered
        onButtonRendered,
        // enableCountDownTimer
        enableCountDownTimer,
        // totalCountDownTime
        totalCountDownTime,
        // currentCountDownTime
        currentCountDownTime
    )
}
