package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.rewardedcountdowntimer

import androidx.compose.animation.core.Animatable
import androidx.compose.runtime.Composable
import androidx.compose.animation.core.LinearEasing
import androidx.compose.animation.core.tween
import androidx.compose.foundation.Canvas
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.shape.CircleShape
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.geometry.Size
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.StrokeCap
import androidx.compose.ui.graphics.drawscope.Stroke
import androidx.compose.ui.layout.onSizeChanged
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.IntSize
import androidx.compose.ui.unit.dp
import androidx.compose.material.Text
import androidx.compose.runtime.saveable.rememberSaveable
import androidx.compose.ui.platform.LocalLifecycleOwner
import androidx.compose.ui.semantics.contentDescription
import androidx.compose.ui.semantics.semantics
import androidx.compose.ui.semantics.testTag
import androidx.compose.ui.text.style.TextAlign
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleEventObserver
import androidx.lifecycle.LifecycleOwner
import com.moloco.sdk.internal.utils.substituteCountDownTimeLeftMacro
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.utils.DEFAULT_FONT_SIZE
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import kotlin.coroutines.cancellation.CancellationException

/**
 * A composable function that displays a circular countdown timer for rewarded content.
 * The timer's text can be customized with the customTimerString - which distinguishes from the RewardedCountDownTimer
 *
 * @param inactiveBarColor The color of the bar that represents inactive time.
 * @param activeBarColor The color of the bar that represents active countdown time.
 * @param modifier [Modifier] to be applied to the timer layout.
 * @param strokeWidth The thickness of the circular timer bar.
 * @param circleSize The diameter of the circular timer.
 * @param onTimerFinish A callback function that is invoked when the timer completes.
 * @param customTimerString A custom string to display as part of the timer.
 * @param lifecycleOwner The [LifecycleOwner] for observing lifecycle changes, with a default value of [LocalLifecycleOwner.current].
 * @param scope The [CoroutineScope] used for launching and managing the countdown timer coroutine, with a default of [rememberCoroutineScope].
 * @param totalCountDownTime The [UInt] used for the total countdown value for the CountDownTimer.
 * @param currentCountDownTime The [UInt] used for managing the countdown and arc animation state for the CountdownTimer.
 */
@Composable
internal fun RewardedCountDownTimerCustom(
    inactiveBarColor: Color,
    activeBarColor: Color,
    modifier: Modifier = Modifier,
    strokeWidth: Dp = 5.dp,
    circleSize: Dp = 30.dp,
    onTimerFinish: () -> Unit,
    customTimerString: String,
    lifecycleOwner: LifecycleOwner = LocalLifecycleOwner.current,
    scope: CoroutineScope = rememberCoroutineScope(),
    totalCountDownTime: UInt,
    currentCountDownTime: UInt
) {

     var size by remember { mutableStateOf(IntSize.Zero) }

    var isTimerRunning by remember(lifecycleOwner) {
        mutableStateOf(true)
    }

    var minProgressRatio by rememberSaveable {
        mutableStateOf(1f)
    }

    val playbackProgress = asProgressRatio(currentCountDownTime, totalCountDownTime).coerceAtMost(minProgressRatio)
    val progressRatio = remember { Animatable(playbackProgress) }
    minProgressRatio = progressRatio.value

    LaunchedEffect(isTimerRunning, currentCountDownTime) {
        if (isTimerRunning && progressRatio.value > 0f) {
            val remainingMillis = (currentCountDownTime * 1000u).toLong()
            try {
                progressRatio.animateTo(
                    targetValue = 0f,
                    animationSpec = tween(
                        durationMillis = remainingMillis.toInt(),
                        easing = LinearEasing
                    )
                )
            } catch (_: CancellationException) {
                // Safe to ignore cancellation as it's expected when the LaunchedEffect is cancelled
                // during recomposition or disposal of the composable
            }
        }
    }

    LaunchedEffect(currentCountDownTime) {
        if (currentCountDownTime == 0u) {
            onTimerFinish()
        }
    }

    DisposableEffect(lifecycleOwner) {
        val observer = LifecycleEventObserver { _, event ->
            when (event) {
                Lifecycle.Event.ON_STOP -> {
                    isTimerRunning = false
                    scope.launch {
                        progressRatio.stop()
                    }
                }
                Lifecycle.Event.ON_START -> {
                    if (progressRatio.value > 0f) {
                        isTimerRunning = true
                    }
                }
                else -> {
                    // Ignore other lifecycle events
                }
            }
        }

        lifecycleOwner.lifecycle.addObserver(observer)

        onDispose {
            lifecycleOwner.lifecycle.removeObserver(observer)
        }
    }

    Row(
        verticalAlignment = Alignment.CenterVertically,
        horizontalArrangement = Arrangement.spacedBy(8.dp),
        modifier = modifier
    ) {
        val textDescription = "custom_countdown_timer_text"
        Text(
            text = customTimerString.substituteCountDownTimeLeftMacro(currentCountDownTime),
            color = activeBarColor,
            maxLines = 2,
            fontSize = DEFAULT_FONT_SIZE,
            textAlign = TextAlign.Start,
            modifier = Modifier.semantics {
                textDescription.let {
                    this.contentDescription = it
                    this.testTag = it
                }
            },
        )

        val boxDescription = "custom_timer_container"
        Box(
            contentAlignment = Alignment.Center,
            modifier = Modifier
                .size(circleSize)
                .padding(2.dp)
                .onSizeChanged {
                    size = it
                }
                .background(Color.White.copy(alpha = 0.9f), shape = CircleShape)
                .semantics {
                    boxDescription.let {
                        this.contentDescription = it
                        this.testTag = it
                    }
                }
        ) {
            Canvas(modifier = Modifier.fillMaxSize()) {
                drawArc(
                    color = inactiveBarColor,
                    startAngle = 360f,
                    sweepAngle = 360f,
                    useCenter = false,
                    size = Size(size.width.toFloat(), size.height.toFloat()),
                    style = Stroke(strokeWidth.toPx(), cap = StrokeCap.Round)
                )
                if (progressRatio.value > 0f) {
                    drawArc(
                        color = activeBarColor,
                        startAngle = 270f,
                        sweepAngle = -360f * progressRatio.value.coerceAtLeast(0f),
                        useCenter = false,
                        size = Size(size.width.toFloat(), size.height.toFloat()),
                        style = Stroke(strokeWidth.toPx(), cap = StrokeCap.Round)
                    )
                }
            }
        }
    }
}
