package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.compose

import android.content.Context
import android.view.View
import androidx.annotation.VisibleForTesting
import androidx.compose.animation.AnimatedVisibility
import androidx.compose.animation.Crossfade
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.BoxScope
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.displayCutoutPadding
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.material.MaterialTheme
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.Shape
import androidx.compose.ui.graphics.painter.Painter
import androidx.compose.ui.platform.ComposeView
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.semantics.contentDescription
import androidx.compose.ui.semantics.semantics
import androidx.compose.ui.semantics.testTag
import androidx.compose.ui.unit.DpSize
import androidx.compose.ui.unit.dp
import com.moloco.sdk.R
import com.moloco.sdk.service_locator.SdkObjectFactory
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ViewVisibilityTracker
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.AdCountdownButton
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.defaultVastAdBadge
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.utils.DEFAULT_AD_BUTTON_PADDING
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.utils.DEFAULT_BUTTON_DP_SIZE
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.utils.DEFAULT_ICON_BUTTON_BACKGROUND_COLOR
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.utils.DEFAULT_ICON_BUTTON_BACKGROUND_SHAPE
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.ExtraOnClick
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.GenericIconButton
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.TrackableButton
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.AdGoNextButton
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.defaultAdCloseCountdownButton
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.modifiers.buttonGlobalPositionModifier
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.theme.Theme
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.utils.defaultButton
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.PlaybackProgress
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.ad.AdViewModel
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService.UserInteraction.Button
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService.UserInteraction.Button.ButtonType
import kotlinx.coroutines.flow.StateFlow

/**
 * @param overrideVastContainerOnClick use this parameter to override default click handling
 * of Vast ad parts: video/linear and endcard/companion parts.
 *
 * null - do not override default click handling.
 *
 * _Important:_ when overridden, the underlying default click handling (VAST click tracking, clickthrough uri link processing) won't be invoked.
 */
@Composable
internal fun VastRenderer(
    adViewModel: AdViewModel,
    modifier: Modifier = Modifier,
    backgroundColor: Color = Color.Black,
    muteButton: MuteButton? = defaultMuteButton(),
    adCloseCountdownButton: AdCountdownButton? = defaultAdCloseCountdownButton(),
    adSkipCountdownButton: AdCountdownButton? = defaultAdSkipCountdownButton(),
    ctaButton: CTAButton? = defaultCTAButton(),
    overrideVastContainerOnClick: OverrideVastContainerOnClick? = null,
    progressBar: ProgressBar? = defaultProgressBar(),
    vastIcon: VastIcon? = defaultVastIcon(),
    playbackControl: PlaybackControl? = null,
    viewVisibilityTracker: ViewVisibilityTracker,
) {
    Box(
        modifier = modifier
            .fillMaxSize()
            .background(backgroundColor)
    ) {
        val currentAdPart by adViewModel.currentAdPart.collectAsState()

        Crossfade(targetState = currentAdPart) { adPart ->
            when (adPart) {
                is AdViewModel.AdPart.Companion -> Companion(
                    adPart.viewModel,
                    overrideVastContainerOnClick?.overrideCompanionOnClick,
                    Modifier.fillMaxSize()
                )
                is AdViewModel.AdPart.Linear -> Linear(
                    adPart.viewModel,
                    overrideVastContainerOnClick?.overrideLinearOnClick,
                    Modifier.fillMaxSize(),
                    muteButton,
                    progressBar,
                    vastIcon,
                    playbackControl,
                    viewVisibilityTracker,
                    onShouldReplay = adViewModel::onReplay
                )
                is AdViewModel.AdPart.DEC -> DEC(
                    viewModel = adPart.viewModel,
                    overrideDECOnClick = overrideVastContainerOnClick?.overrideDECOnClick,
                    vastIcon,
                    Modifier.fillMaxSize()
                )
                is AdViewModel.AdPart.Mraid -> {/* NO OP */ }
                null -> {}
            }
        }

        AdGoNextButton(
            adViewModel,
            currentAdPart,
            adSkipCountdownButton,
            adCloseCountdownButton
        )

        ctaButton?.let {
            val ctaAvailable by adViewModel.ctaAvailable.collectAsState()
            it(
                ctaAvailable,
                adViewModel.currentAdPart,
                adViewModel::onButtonRendered,
                adViewModel::onCTA
            )
        }

        defaultVastAdBadge(
            modifier = Modifier
                .align(Alignment.BottomStart)
                .padding(12.dp)
        )(adViewModel::onButtonRendered, adViewModel.currentAdPart)
    }
}

/**
 * Use the instance of this class to override default click handling in _VastRenderer()_ implementation.
 *
 * _Important:_ when overridden, the underlying default click handling (VAST click tracking, clickthrough uri link processing) won't be invoked.
 */
class OverrideVastContainerOnClick(
    /**
     * overrides clicks on linear/video part of the ad; when null -- do not override.
     */
    val overrideLinearOnClick: (() -> Unit)? = null,
    /**
     * overrides clicks on companion/endcard part of the ad; when null -- do not override.
     */
    val overrideCompanionOnClick: (() -> Unit)? = null,

    /**
     * overrides clicks on DEC part of the ad; when null -- do not override.
     */
    val overrideDECOnClick: (() -> Unit)? = null,
)

typealias VastRenderer = (context: Context, adViewModel: AdViewModel) -> View

/**
 * @param overrideVastContainerOnClick use this parameter to override default click handling
 * of Vast ad parts: video/linear and endcard/companion parts.
 *
 * null - do not override default click handling.
 *
 * _Important:_ when overridden, the underlying default click handling (VAST click tracking, clickthrough uri link processing) won't be invoked.
 */
internal fun defaultVastRenderer(
    backgroundColor: Color = Color.Black,
    muteButton: @Composable () -> MuteButton? = { defaultMuteButton() },
    adCloseCountdownButton: @Composable () -> AdCountdownButton? = { defaultAdCloseCountdownButton() },
    adSkipCountdownButton: @Composable () -> AdCountdownButton? = { defaultAdSkipCountdownButton() },
    ctaButton: @Composable () -> CTAButton? = { defaultCTAButton() },
    overrideVastContainerOnClick: OverrideVastContainerOnClick? = null,
    progressBar: @Composable () -> ProgressBar? = { defaultProgressBar() },
    vastIcon: @Composable () -> VastIcon? = { defaultVastIcon() },
    playbackControl: @Composable () -> PlaybackControl? = { null },
    viewVisibilityTracker: ViewVisibilityTracker = SdkObjectFactory.Miscellaneous.viewVisibilityTrackerFactory,
): VastRenderer = { context, adViewModel ->
    ComposeView(context).apply {
        @VisibleForTesting(otherwise = VisibleForTesting.NONE)
        id = R.id.moloco_fullscreen_ad_view_id

        setContent {
            Theme {
                VastRenderer(
                    adViewModel = adViewModel,
                    backgroundColor = backgroundColor,
                    muteButton = muteButton(),
                    adCloseCountdownButton = adCloseCountdownButton(),
                    adSkipCountdownButton = adSkipCountdownButton(),
                    ctaButton = ctaButton(),
                    overrideVastContainerOnClick = overrideVastContainerOnClick,
                    progressBar = progressBar(),
                    vastIcon = vastIcon(),
                    playbackControl = playbackControl(),
                    viewVisibilityTracker = viewVisibilityTracker,
                )
            }
        }
    }
}

typealias MuteButton = @Composable BoxScope.(
    isPlayerPlaying: Boolean, // Indicates whether the player is playing (video maybe buffering)
    mute: Boolean,
    onButtonReplaced: (Button, ButtonType) -> Unit,
    onMuteChange: (Boolean) -> Unit
) -> Unit

@Composable
fun defaultMuteButton(
    size: DpSize = DEFAULT_BUTTON_DP_SIZE,
    iconSize: DpSize = size,
    backgroundShape: Shape = DEFAULT_ICON_BUTTON_BACKGROUND_SHAPE,
    backgroundColor: Color = DEFAULT_ICON_BUTTON_BACKGROUND_COLOR,
    alignment: Alignment = Alignment.TopStart,
    padding: PaddingValues = PaddingValues(DEFAULT_AD_BUTTON_PADDING),
    color: Color = MaterialTheme.colors.primary,
    muteIcon: Painter = painterResource(id = R.drawable.ic_round_volume_off_24),
    unmuteIcon: Painter = painterResource(id = R.drawable.ic_round_volume_up_24),
    extraOnClick: ExtraOnClick? = null
): MuteButton = { isPlayerPlaying, mute, onButtonReplaced, onMuteChange ->

    // For mute button tracking, we have a custom state (rather than using the TrackableButton)
    // composable because mute button has dual states that need to be reported individually
    // It's not merely reporting it's visibility/position/size but also it's state (mute/unmute)
    var muteButtonLayout by remember {
        mutableStateOf(defaultButton(ButtonType.MUTE))
    }

    AnimatedVisibility(
        visible = isPlayerPlaying,
        modifier = Modifier
            .align(alignment)
            .displayCutoutPadding()
            .padding(padding)
    ) {
        val buttonDescription = "mute_button"
        GenericIconButton(
            enabled = isPlayerPlaying,
            painter = if (mute) muteIcon else unmuteIcon,
            modifier = buttonGlobalPositionModifier(Modifier, muteButtonLayout) {
                muteButtonLayout = it
                onButtonReplaced(
                    muteButtonLayout,
                    if (mute) ButtonType.MUTE else ButtonType.UNMUTE
                )
            }.semantics {
                buttonDescription.let {
                    this.contentDescription = it
                    this.testTag = it
                }
            },
            onClick = {
                onMuteChange(!mute)
                // We need to change the layout here on click because `globalPosition` does not
                // change when icon (painter) changes. Thus we handle that scenario specifically
                muteButtonLayout = Button(if (mute) ButtonType.MUTE else ButtonType.UNMUTE, position = muteButtonLayout.position, size = muteButtonLayout.size)
                onButtonReplaced(
                    muteButtonLayout,
                    if (mute) ButtonType.UNMUTE else ButtonType.MUTE
                )
                extraOnClick?.invoke()
            },
            contentDescription = "mute/unmute",
            tint = color,
            size = size,
            iconSize = iconSize,
            backgroundShape = backgroundShape,
            backgroundColor = backgroundColor
        )
    }
}

typealias CTAButton = @Composable BoxScope.(
    ctaAvailable: Boolean,
    currentAdViewPart: StateFlow<AdViewModel.AdPart?>,
    onButtonRendered: (Button) -> Unit,
    onCTA: () -> Unit
) -> Unit

@Composable
fun defaultCTAButton(
    alignment: Alignment = Alignment.BottomEnd,
    padding: PaddingValues = PaddingValues(DEFAULT_AD_BUTTON_PADDING),
    color: Color = MaterialTheme.colors.primary,
    text: String = stringResource(id = R.string.com_moloco_sdk_xenoss_player_learn_more),
    imageUri: String? = null,
    extraOnClick: ExtraOnClick? = null
): CTAButton = { ctaAvailable, currentAdPartFlow, onButtonRendered, onCTA ->
    val currentAdPart by currentAdPartFlow.collectAsState()

    AnimatedVisibility(
        visible = ctaAvailable,
        modifier = Modifier
            .align(alignment)
            .displayCutoutPadding()
            .padding(padding)
    ) {
        // We show the same Vast CTA between linear and companion, however
        // because linear and companion have different view models, they have different
        // states of which buttons are shown. The ButtonType.CTA is de-duped at jetpack compose
        // to avoid repeatedly getting notification on a recompose but no changes to
        // the position/sizing. The side effect is that when we go from Linear -> Companion
        // we have no way to trigger a button rendered callback to the companion view model.
        // Thus we trigger a recomposition here by rendering a different instance of the
        // composable. This is not ideal but due to how JP compose spams us with button modifiers
        // we need to de-dupe to avoid clobbering the main thread and thus to workaround the de-dupe
        // we need to re-render one additional time. Visually this has no impact
        when (currentAdPart) {
            is AdViewModel.AdPart.Companion -> {
                TrackableButton(
                    buttonType = ButtonType.CTA,
                    onButtonRendered = onButtonRendered
                ) {
                    VastCTA(
                        modifier = it,
                        onClick = {
                            onCTA()
                            extraOnClick?.invoke()
                        },
                        text = text,
                        color = color,
                        imageUri = imageUri
                    )
                }
            }
            is AdViewModel.AdPart.Linear -> {
                TrackableButton(
                    buttonType = ButtonType.CTA,
                    onButtonRendered = onButtonRendered
                ) {
                    VastCTA(
                        modifier = it,
                        onClick = {
                            onCTA()
                            extraOnClick?.invoke()
                        },
                        text = text,
                        color = color,
                        imageUri = imageUri
                    )
                }
            }

            is AdViewModel.AdPart.DEC -> {
                /* NO OP */
            }
            is AdViewModel.AdPart.Mraid -> {/* NO OP */ }
            null -> { /* NO OP */ }
        }
    }
}

typealias ProgressBar = @Composable BoxScope.(
    isPlaying: Boolean,
    progress: PlaybackProgress
) -> Unit

@Composable
fun defaultProgressBar(
    alignment: Alignment = Alignment.BottomCenter,
    padding: PaddingValues = PaddingValues(0.dp),
    color: Color = MaterialTheme.colors.primary,
): ProgressBar = { isPlaying, progress ->
    val shouldShow = isPlaying || (progress is PlaybackProgress.Position && progress.currentPositionMillis > 0)
    AnimatedVisibility(
        visible = shouldShow,
        modifier = Modifier
            .fillMaxWidth()
            .align(alignment)
            .padding(padding)
    ) {
        VastProgressBar(
            isPlaying = isPlaying,
            progress = progress,
            color = color
        )
    }
}
