package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.compose

import android.view.ViewGroup
import android.widget.FrameLayout
import androidx.compose.runtime.Composable
import androidx.compose.runtime.DisposableEffect
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.rememberUpdatedState
import androidx.compose.runtime.saveable.rememberSaveable
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.platform.LocalLifecycleOwner
import androidx.compose.ui.viewinterop.AndroidView
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.service_locator.SdkObjectFactory
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ViewVisibilityTracker
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.VastAdShowError
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.PlaybackProgress
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.ValueWrapper
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.videoplayer.SimplifiedExoPlayer
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.videoplayer.VideoPlayer
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.videoplayer.VisibilityAwareVideoPlayer
import kotlinx.coroutines.flow.filterNotNull
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.withContext

// Autoplays video, since that's good enough for my use cases.
@Composable
fun VastVideoPlayer(
    uri: String,
    isStreamingEnabled: Boolean,
    play: ValueWrapper<Boolean>,
    seekToMillis: ValueWrapper<Long>,
    isMute: Boolean,
    isPlaying: (Boolean) -> Unit,
    isVisible: (Boolean) -> Unit,
    viewVisibilityTracker: ViewVisibilityTracker,
    onProgressChanged: (PlaybackProgress) -> Unit,
    onError: (VastAdShowError) -> Unit,
    modifier: Modifier = Modifier
) {
    val context = LocalContext.current
    val lifecycle = LocalLifecycleOwner.current.lifecycle

    val videoPlayer: VideoPlayer = remember(context, isStreamingEnabled, lifecycle) {
        val simplifiedExoPlayer = SimplifiedExoPlayer(
            context = context,
            isProgressivePlayingEnabled = isStreamingEnabled,
            mediaCacheRepository = SdkObjectFactory.Media.mediaCacheRepository,
            lifecycle = lifecycle
        )
        VisibilityAwareVideoPlayer(simplifiedExoPlayer, viewVisibilityTracker)
    }

    val scope = rememberCoroutineScope { DispatcherMain }

    // TODO. Not necessary now, but might make sense reconsidering moving to separate DisposableEffects
    //  with respective Flow subscriptions in order to support event propagation right after subscription.
    //  Current implementation will not fire last event value to callbacks if the latter ones changed.
    val currentIsPlaying by rememberUpdatedState(isPlaying)
    val currentIsVisible by rememberUpdatedState(newValue = isVisible)
    val hasMore = rememberSaveable {
        mutableStateOf<Boolean>(true)
    }
    val isPlayingRightNow = rememberSaveable {
        mutableStateOf<Boolean>(true)
    }
    val currentOnProgressChanged by rememberUpdatedState(onProgressChanged)
    val currentOnError by rememberUpdatedState(onError)

    DisposableEffect(videoPlayer) {
        val jobs = listOf(
            videoPlayer.isPlaying.onEach {
                currentIsPlaying(it.isPlaying)
                isPlayingRightNow.value = it.isPlaying
                hasMore.value = it.hasMore
                currentIsVisible(it.isVisible)
                videoPlayer.playerView?.keepScreenOn = it.hasMore
            }.launchIn(scope),

            videoPlayer.playbackProgress.onEach {
                currentOnProgressChanged(it)
            }.launchIn(scope),

            videoPlayer.lastError.filterNotNull().onEach {
                currentOnError(it)
            }.launchIn(scope)
        )

        onDispose {
            jobs.onEach { it.cancel() }
            videoPlayer.destroy()

            // TODO. Test this solution. It's called from side-effect, so it should be safe and correct.
            currentIsPlaying(false)
        }
    }

    LaunchedEffect(videoPlayer, uri, seekToMillis) {
        withContext(DispatcherMain) {
            with(videoPlayer) {
                uriSource = uri
                seekTo(seekToMillis.value)
                // TODO. Workaround based on knowledge of implementation details.
                //  No API constraints/conventions used.
                // UriSource change resets latest play/pause command to pause.
                // We need to explicitly call play/pause again.
                playOrPause(play)
            }
        }
    }

    LaunchedEffect(videoPlayer, play) {
        withContext(DispatcherMain) {
            with(videoPlayer) {
                playOrPause(play)
            }
        }
    }

    LaunchedEffect(videoPlayer, isMute) {
        withContext(DispatcherMain) {
            videoPlayer.isMute = isMute
        }
    }

    videoPlayer.playerView?.let { playerView ->
        AndroidView(
            modifier = modifier,
            // Factory is called only once.
            factory = {
                FrameLayout(it).also { fl ->
                    fl.addView(playerView, ViewGroup.LayoutParams(
                        ViewGroup.LayoutParams.MATCH_PARENT,
                        ViewGroup.LayoutParams.MATCH_PARENT
                    ))
                }
            }
        )
    }
}

private fun VideoPlayer.playOrPause(play: ValueWrapper<Boolean>) {
    if (play.value) play() else pause()
}

private val DispatcherMain = DispatcherProvider().main
