package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.compose.touch

import androidx.compose.foundation.gestures.GestureCancellationException
import androidx.compose.foundation.gestures.PressGestureScope
import androidx.compose.foundation.gestures.awaitFirstDown
import androidx.compose.foundation.gestures.forEachGesture
import androidx.compose.ui.geometry.Offset
import androidx.compose.ui.input.pointer.AwaitPointerEventScope
import androidx.compose.ui.input.pointer.PointerEventPass
import androidx.compose.ui.input.pointer.PointerInputChange
import androidx.compose.ui.input.pointer.PointerInputScope
import androidx.compose.ui.input.pointer.changedToUp
import androidx.compose.ui.input.pointer.isOutOfBounds
import androidx.compose.ui.unit.Density
import kotlinx.coroutines.coroutineScope
import kotlinx.coroutines.sync.Mutex

/**
 * The last tap touch position when a tap is detected
 */
internal typealias LastTouchTap = Offset

/**
 * The first tap touch position when a tap is detected
 */
internal typealias FirstTouchTap = Offset

/**
 * Callback for when a tap is detected but not consumed
 */
internal typealias onTapUnconsumed = (FirstTouchTap, LastTouchTap) -> Unit

/**
 * Captures a tap but does not consume it. This is particularly useful if parent composables
 * want to capture a tap but not override a child component's click / touch handlers.
 *
 * EG usages are for Analytics, Logging etc
 */
internal suspend fun PointerInputScope.detectTapUnconsumed(onTap: onTapUnconsumed) {
    val pressScope = PressGestureScopeImpl2(this)
    forEachGesture {
        coroutineScope {
            pressScope.reset()
            awaitPointerEventScope {
                val down = awaitFirstDown(requireUnconsumed = false).also {
                    if (it.pressed != it.previousPressed) it.consume()
                }
                val up = waitForUpOrCancellationInitial()
                if (up == null) {
                    pressScope.cancel()
                } else {
                    pressScope.release()
                    onTap(down.position, up.position)
                }
            }
        }
    }
}

private suspend fun AwaitPointerEventScope.waitForUpOrCancellationInitial(): PointerInputChange? {
    while (true) {
        val event = awaitPointerEvent(PointerEventPass.Initial)
        if (event.changes.fastAll { it.changedToUp() }) {
            return event.changes[0]
        }
        if (event.changes.fastAny {
                it.isConsumed ||
                    it.isOutOfBounds(size, extendedTouchPadding)
            }
        ) {
            return null
        }
        // Check for cancel by position consumption.
        // We can look on the Final pass of the existing
        // pointer event because it comes after the Main
        // pass we checked above.
        val consumeCheck = awaitPointerEvent(PointerEventPass.Final)
        if (consumeCheck.changes.fastAny {
                it.isConsumed
            }
        ) {
            return null
        }
    }
}

/**
 * [detectTapGestures]'s implementation of [PressGestureScope].
 */
private class PressGestureScopeImpl2(
    density: Density
) : PressGestureScope, Density by density {
    private var isReleased = false
    private var isCanceled = false
    private val mutex = Mutex(locked = false)

    /**
     * Called when a gesture has been canceled.
     */
    fun cancel() {
        isCanceled = true
        mutex.unlock()
    }

    /**
     * Called when all pointers are up.
     */
    fun release() {
        isReleased = true
        mutex.unlock()
    }

    /**
     * Called when a new gesture has started.
     */
    fun reset() {
        mutex.tryLock() // If tryAwaitRelease wasn't called, this will be unlocked.
        isReleased = false
        isCanceled = false
    }

    override suspend fun awaitRelease() {
        if (!tryAwaitRelease()) {
            throw GestureCancellationException("The press gesture was canceled.")
        }
    }

    override suspend fun tryAwaitRelease(): Boolean {
        if (!isReleased && !isCanceled) {
            mutex.lock()
        }
        return isReleased
    }
}

@Suppress("BanInlineOptIn")
private inline fun <T> List<T>.fastForEach(action: (T) -> Unit) {
    for (index in indices) {
        val item = get(index)
        action(item)
    }
}

@Suppress("BanInlineOptIn")
private inline fun <T> List<T>.fastAll(predicate: (T) -> Boolean): Boolean {
    fastForEach { if (!predicate(it)) return false }
    return true
}

@Suppress("BanInlineOptIn")
private inline fun <T> List<T>.fastAny(predicate: (T) -> Boolean): Boolean {
    fastForEach { if (predicate(it)) return true }
    return false
}
