package com.moloco.sdk.internal.mediators

import kotlin.time.Duration
import kotlin.time.DurationUnit
import kotlin.time.toDuration

// Mediators are case sensitive
internal enum class Mediators {
    MAX,
    LevelPlay,
    X3MADS,
    AdMob,
    Custom_MAX,
    Custom_LevelPlay,
    Custom_AdMob,
}


/**
 * Calculates the timeout duration for loading an interstitial ad for the given mediator.
 *
 * @param mediator The mediator for which the timeout duration is to be calculated. Can be null.
 * @return The duration for the ad load timeout.
 */
// TODO: Make this configurable - https://mlc.atlassian.net/browse/SDK-1784
internal fun fullScreenAdLoadTimeout(mediator: String?): Duration {
    return when (mediator) {
        /**
         * For those wondering how we came to this number...
         * It took planning and guile. Some sleepless nights and the moto, try hard or die trying.
         * Finally we arrived at the conclusion:
         * Applovin MAX has a 30 seconds timeout. We subtract 4 second to avoid timeout race condition.
         */
        Mediators.MAX.name -> 26.toDuration(DurationUnit.SECONDS)

        // AdMob has a timeout of 30-45 seconds. We go with the most aggressive timeout.
        Mediators.AdMob.name -> 26.toDuration(DurationUnit.SECONDS)

        // Unity LevelPlay has a 27 second timeout. We subtract 1 second to avoid a timeout race condition.
        Mediators.LevelPlay.name -> 26.toDuration(DurationUnit.SECONDS)

        // By default we set the timeout to 60 seconds so we have a higher chance of getting an ad.
        // Custom waterfall is now not supported in Moloco SDK. This is more for publisher in house mediation
        // https://www.notion.so/moloco/Publisher-In-House-Mediation-Tech-Proposal-184cdb35133680ca96ecc9dded1fe79e
        else -> 60.toDuration(DurationUnit.SECONDS)
    }
}

/**
 * Calculates the timeout duration for loading a banner ad for the given mediator.
 *
 * @param mediator The mediator for which the timeout duration is to be calculated. Can be null.
 * @return The duration for the ad load timeout.
 */
internal fun bannerAdLoadTimeout(mediator: String?): Duration {
    return when (mediator) {
        /**
         * For those wondering how we came to this number...
         * It took planning and guile. Some sleepless nights and the moto, try hard or die trying.
         * Finally we arrived at the conclusion:
         * Applovin MAX has a 10 seconds timeout. We subtract 2 second to avoid timeout race condition.
         * Unity LevelPlay a 10 seconds timeout. We subtract 2 second to avoid timeout race condition.
         */
        Mediators.MAX.name -> 8.toDuration(DurationUnit.SECONDS)
        Mediators.LevelPlay.name -> 8.toDuration(DurationUnit.SECONDS)
        Mediators.AdMob.name -> 8.toDuration(DurationUnit.SECONDS)

        // By default we set the timeout to 60 seconds so we have a higher chance of getting an ad.
        // Custom waterfall is now not supported in Moloco SDK. This is more for publisher in house mediation
        // https://www.notion.so/moloco/Publisher-In-House-Mediation-Tech-Proposal-184cdb35133680ca96ecc9dded1fe79e
        else -> 60.toDuration(DurationUnit.SECONDS)
    }
}

/**
 * Calculates the timeout duration for loading a banner ad for the given mediator.
 *
 * @param mediator The mediator for which the timeout duration is to be calculated. Can be null.
 * @return The duration for the ad load timeout.
 */
internal fun nativeAdLoadTimeout(mediator: String?): Duration {
    return when (mediator) {
        /**
         * Max uses 10 seconds for timeout. We subtract 2 second to avoid timeout race condition.
         */
        Mediators.MAX.name -> 8.toDuration(DurationUnit.SECONDS)
        /**
         * Admob uses 60 seconds for timeout. We subtract 2 second to avoid timeout race condition.
         */
        Mediators.AdMob.name -> 58.toDuration(DurationUnit.SECONDS)

        // By default we set the timeout to 60 seconds so we have a higher chance of getting an ad.
        // Custom waterfall is now not supported in Moloco SDK. This is more for publisher in house mediation
        // https://www.notion.so/moloco/Publisher-In-House-Mediation-Tech-Proposal-184cdb35133680ca96ecc9dded1fe79e
        else -> 60.toDuration(DurationUnit.SECONDS)
    }
}
