package com.moloco.sdk.internal.publisher

import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.publisher.AdFormatType
import com.moloco.sdk.publisher.Moloco
import kotlin.time.Duration
import kotlin.time.DurationUnit
import kotlin.time.toDuration

/**
 * Encapsulates the logic for tracking ad creation and loading times,
 * and calculating the remaining timeout based on those times.
 */
internal class AdCreateLoadTimeoutManager(private val adFormatType: AdFormatType, private val mediationTimeoutSeconds: Duration) : CreateAdObjectTime {
    /**
     * The timestamp representing the start time when the ad object creation begins.
     *
     * This variable is set at the beginning of the `Ad.create()` call, marking the point when the ad creation process starts.
     */
    override var createAdObjectStartTime: Long = 0

    /**
     * Calculates the remaining timeout for the ad load process based on the time taken to create the ad object.
     *
     * This function subtracts the duration of the ad creation process from the mediation timeout to determine
     * how much time is left before the ad load process times out.
     *
     * @param loadStartTimeMs The timestamp representing the start time when the ad object begins loading.
     *
     * @return The remaining timeout as a `Duration`, after subtracting the ad creation time from the mediation timeout.
     */
    fun calculateTimeout(loadStartTimeMs: Long): Duration {
        // Ensure that all maths is on milliseconds unit
        val createAdDuration = (loadStartTimeMs - createAdObjectStartTime)
        val remainingTimeout = (mediationTimeoutSeconds.inWholeMilliseconds - createAdDuration).toDuration(DurationUnit.MILLISECONDS)
        MolocoLogger.debug(TAG, "$adFormatType timeout: $mediationTimeoutSeconds , create ad duration: $createAdDuration ms (createTime: $createAdObjectStartTime ms, loadStartTime: $loadStartTimeMs ms). Return value: $remainingTimeout")

        return remainingTimeout
    }
    companion object{
        private const val TAG = "AdCreateLoadTimeoutManager"
    }
}

/**
 * Used for measuring time it take to create an ad object.
 * Ex.: [Moloco.createBanner] -> time to create
 */
internal interface CreateAdObjectTime {
    var createAdObjectStartTime : Long
}
