package com.moloco.sdk.internal.publisher

import androidx.annotation.VisibleForTesting
import com.moloco.sdk.acm.CountEvent
import com.moloco.sdk.acm.TimerEvent
import com.moloco.sdk.internal.MolocoInternalAdError
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.SdkEventUrlTracker
import com.moloco.sdk.internal.client_metrics_data.AcmCount
import com.moloco.sdk.internal.client_metrics_data.AcmTag
import com.moloco.sdk.internal.ortb.model.SdkEvents
import com.moloco.sdk.publisher.AdFormatType
import com.moloco.sdk.publisher.AdLoad
import com.moloco.sdk.publisher.MolocoAd
import com.moloco.sdk.acm.AndroidClientMetrics as acm

/**
 * Wraps the origin listener and fires url tracking event upon calling the respective listener's callback.
 */
internal fun AdLoadListenerTracker(
    originListener: AdLoad.Listener?,
    acmLoadTimerEvent: TimerEvent,
    adFormatType: AdFormatType
): InternalAdLoadListener = AdLoadListenerTrackerImpl(
    originListener,
    SdkEventUrlTracker(),
    acmLoadTimerEvent,
    adFormatType
)

@VisibleForTesting
internal class AdLoadListenerTrackerImpl(
    private val originListener: AdLoad.Listener?,
    private val sdkEventUrlTracker: SdkEventUrlTracker,
    private val acmLoadTimerEvent: TimerEvent,
    private val adFormatType: AdFormatType
) : InternalAdLoadListener {
    override fun onAdLoadStarted(molocoAd: MolocoAd, timestamp: Long, sdkEvents: SdkEvents?) {
        MolocoLogger.info(TAG, "onAdLoadStarted: $molocoAd, $timestamp")
        sdkEvents?.onAdLoadStart?.let { url ->
            sdkEventUrlTracker(url, timestamp)
        }
    }

    override fun onAdLoadSuccess(molocoAd: MolocoAd, sdkEvents: SdkEvents?) {
        MolocoLogger.info(TAG, "onAdLoadSuccess: $molocoAd")
        sdkEvents?.onAdLoadSuccess?.let { url ->
            sdkEventUrlTracker(url, System.currentTimeMillis())
        }

        acm.recordTimerEvent(
            acmLoadTimerEvent
                .withTag(AcmTag.Result.tagName, "success")
                .withTag(AcmTag.AdType.tagName, adFormatType.name.lowercase())
        )

        acm.recordCountEvent(
            CountEvent(AcmCount.LoadAdSuccess.eventName)
                .withTag(AcmTag.AdType.tagName, adFormatType.name.lowercase())
        )

        originListener?.onAdLoadSuccess(molocoAd)
    }

    override fun onAdLoadFailed(internalError: MolocoInternalAdError, sdkEvents: SdkEvents?) {
        MolocoLogger.info(TAG, "onAdLoadFailed: $internalError")
        sdkEvents?.onAdLoadFailed?.let { url ->
            sdkEventUrlTracker(url, System.currentTimeMillis(), error = internalError)
        }

        acm.recordTimerEvent(
            acmLoadTimerEvent
                .withTag(AcmTag.Result.tagName, "failure")
                .withTag(AcmTag.Reason.tagName, internalError.subErrorType.metricsRepresentation)
                .withTag(AcmTag.AdType.tagName, adFormatType.name.lowercase())
        )

        acm.recordCountEvent(
            CountEvent(AcmCount.LoadAdFailed.eventName)
                .withTag("network", internalError.molocoAdError.networkName)
                .withTag(AcmTag.Reason.tagName, internalError.subErrorType.metricsRepresentation)
                .withTag(AcmTag.AdType.tagName, adFormatType.name.lowercase())
        )

        originListener?.onAdLoadFailed(internalError.molocoAdError)
    }

    companion object{
        private const val TAG = "AdLoadListenerTrackerImpl"
    }
}
