package com.moloco.sdk.internal.publisher

import androidx.annotation.VisibleForTesting
import com.moloco.sdk.internal.InternalSDKErrorSubType
import com.moloco.sdk.internal.MolocoInternalAdError
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.Result
import com.moloco.sdk.internal.createInternalAdErrorInfo
import com.moloco.sdk.internal.ortb.BidResponseParser
import com.moloco.sdk.internal.ortb.model.Bid
import com.moloco.sdk.publisher.MolocoAdError.ErrorType


/**
 * A class responsible for loading and parsing bid responses for ads.
 *
 * The `BidLoader` class orchestrates the bid parsing process by first processing the raw bid response
 * using a list of preprocessors, and then parsing the processed bid response using a `BidResponseParser`.
 * It also records various events using ACM.
 *
 * The parsing process involves multiple stages:
 * 1. Processing the raw bid response using a preprocessor.
 * 2. Parsing the processed bid response into a `Bid` object.
 * 3. Handling any errors that may occur during these processes by returning appropriate `Result.Failure`.
 */
internal class BidLoader(
    private val bidResponseParser: BidResponseParser,
    private val bidProcessor: BidProcessor,
) {
    suspend fun parse(adUnitId: String, rawBidResponseJson: String): Result<Bid, MolocoInternalAdError> {
        MolocoLogger.debug(TAG, "parse() called with bidResponseJson: $rawBidResponseJson")

        val processedBidResponseJson = processBidResponse(rawBidResponseJson) ?: kotlin.run {
            val error = createInternalAdErrorInfo(
                adUnitId,
                ErrorType.AD_BID_PARSE_ERROR,
                InternalSDKErrorSubType.BID_LOAD_ERROR_CANNOT_PROCESS_BID_RESPONSE)
            return Result.Failure(error)
        }

        MolocoLogger.info(
            TAG,
            "Processed the bidResponse, proceeding with parsing it."
        )

        return parseBidResponse(processedBidResponseJson, adUnitId)
    }

    /**
     * Pre-processes the bid response for the current mediation. If the mediation is not recognized,
     * the original bid response is returned.
     *
     * In case an error happens in any of the [BidProcessor.process] implementations,
     * then this returns null.
     */
    @VisibleForTesting
    internal suspend fun processBidResponse(bidResponseJson: String): String? {
        val proccessedBidResponse = bidProcessor.process(bidResponseJson)
        if (proccessedBidResponse != null) {
            MolocoLogger.warn(
                TAG, "Found no pre-preprocessor for the current mediation. Returning the original bid response.")
            return proccessedBidResponse
        }
        // If no preprocessor is found, return the original bid response
        return bidResponseJson
    }

    private suspend fun parseBidResponse(
        bidResponseJson: String,
        adUnitId: String,
    ): Result<Bid, MolocoInternalAdError> = bidResponseParser.invoke(bidResponseJson).let {
        when (it) {
            is Result.Failure -> {
                MolocoLogger.error(TAG, "parseBidResponse failed to parse BID json string.", it.value)

                val error = createInternalAdErrorInfo(
                    adUnitId,
                    ErrorType.AD_BID_PARSE_ERROR,
                    InternalSDKErrorSubType.BID_LOAD_ERROR_CANNOT_PARSE_BID_RESPONSE
                )
                Result.Failure(error)
            }

            is Result.Success -> {
                Result.Success(it.value.seatBid[0].bid[0])
            }
        }
    }

    companion object {
        private const val TAG = "BidLoader"
    }
}
