package com.moloco.sdk.internal.publisher

import com.moloco.sdk.acm.CountEvent
import com.moloco.sdk.internal.BUrlTracker
import com.moloco.sdk.internal.MolocoInternalAdError
import com.moloco.sdk.internal.SdkEventUrlTracker
import com.moloco.sdk.internal.client_metrics_data.AcmCount
import com.moloco.sdk.internal.client_metrics_data.AcmTag
import com.moloco.sdk.internal.ortb.model.SdkEvents
import com.moloco.sdk.internal.scheduling.GlobalScopes
import com.moloco.sdk.internal.services.AnalyticsApplicationLifecycleTracker
import com.moloco.sdk.publisher.AdFormatType
import com.moloco.sdk.publisher.AdShowListener
import com.moloco.sdk.publisher.MolocoAd
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService
import kotlinx.coroutines.launch
import com.moloco.sdk.acm.AndroidClientMetrics as acm

/**
 * Internal listener interface for ad show events.
 */
internal interface InternalAdShowListener {
    /**
     * Called when an ad starts displaying. Impression can be recorded.
     *
     * @param molocoAd The [MolocoAd] instance that provides information about the ad
     */
    fun onAdShowSuccess(molocoAd: MolocoAd)

    /**
     * Called when an error occurs during ad display.
     *
     * @param internalError The [MolocoInternalAdError] instance that provides information about the error
     */
    fun onAdShowFailed(internalError: MolocoInternalAdError)

    /**
     * Called when an ad is hidden.
     *
     * @param molocoAd The [MolocoAd] instance that provides information about the ad
     */
    fun onAdHidden(molocoAd: MolocoAd)

    /**
     * Called when an ad is clicked.
     *
     * @param molocoAd The [MolocoAd] instance that provides information about the ad
     */
    fun onAdClicked(molocoAd: MolocoAd)
}

/**
 * Wraps the origin listener and fires url tracking event upon calling the respective listener's callback.
 */
internal fun InternalAdShowListenerTracker(
    originListener: AdShowListener?,
    appLifecycleTrackerService: AnalyticsApplicationLifecycleTracker,
    customUserEventBuilderService: CustomUserEventBuilderService,
    provideSdkEvents: () -> SdkEvents?,  // TODO: Make it non null. https://mlc.atlassian.net/browse/SDK-3123
    provideBUrlData: () -> BUrlData?,
    sdkEventUrlTracker: SdkEventUrlTracker = SdkEventUrlTracker(),
    bUrlTracker: BUrlTracker = BUrlTracker(),
    adType: AdFormatType
): InternalAdShowListener = InternalAdShowListenerImpl(
    originListener,
    appLifecycleTrackerService,
    customUserEventBuilderService,
    provideSdkEvents,
    provideBUrlData,
    sdkEventUrlTracker,
    bUrlTracker,
    adType
)

internal class InternalAdShowListenerImpl(
    private val originListener: AdShowListener?,
    private val appLifecycleTrackerService: AnalyticsApplicationLifecycleTracker,
    private val customUserEventBuilderService: CustomUserEventBuilderService,
    private val provideSdkEvents: () -> SdkEvents?,
    private val provideBUrlData: () -> BUrlData?,
    private val sdkEventUrlTracker: SdkEventUrlTracker,
    private val bUrlTracker: BUrlTracker,
    private val adType: AdFormatType
) : InternalAdShowListener {

    override fun onAdShowSuccess(molocoAd: MolocoAd) {
        provideSdkEvents()?.onAdShowSuccess?.let { url ->
            sdkEventUrlTracker(url, System.currentTimeMillis())
        }

        provideBUrlData()?.let {
            val impressionTimestamp = System.currentTimeMillis()
            GlobalScopes.globalIOScope.launch {
                val modifiedUrl = customUserEventBuilderService
                    .userAdInteractionExtAsQueryParameter(
                        impressionTimestamp,
                        CustomUserEventBuilderService.UserInteraction.Impression,
                        it.burl
                    )
                bUrlTracker(modifiedUrl)
            }
        }

        acm.recordCountEvent(CountEvent(AcmCount.ShowAdSuccess.eventName)
            .withTag(AcmTag.AdType.tagName, adType.name.lowercase())
        )
        originListener?.onAdShowSuccess(molocoAd)
    }

    override fun onAdShowFailed(internalError: MolocoInternalAdError) {
        provideSdkEvents()?.onAdShowFailed?.let { url ->
            sdkEventUrlTracker(url, System.currentTimeMillis(), error = internalError)
        }

        acm.recordCountEvent(CountEvent(AcmCount.ShowAdFailed.eventName)
            .withTag(AcmTag.AdType.tagName, adType.name.lowercase())
            .withTag(AcmTag.Reason.tagName, "${internalError.molocoAdError.errorType}")
        )
        originListener?.onAdShowFailed(internalError.molocoAdError)
    }

    override fun onAdHidden(molocoAd: MolocoAd) {
        provideSdkEvents()?.onAdHidden?.let { url ->
            sdkEventUrlTracker(url, System.currentTimeMillis())
        }

        originListener?.onAdHidden(molocoAd)
    }

    override fun onAdClicked(molocoAd: MolocoAd) {
        // When a click on an ad is triggered, register for analytics publishing to record
        // the next background/foreground of the application
        appLifecycleTrackerService.trackNextBackgroundForeground()

        provideSdkEvents()?.onAdClicked?.let { url ->
            sdkEventUrlTracker(url, System.currentTimeMillis())
        }

        acm.recordCountEvent(CountEvent(AcmCount.AdClicked.eventName)
            .withTag(AcmTag.AdType.tagName, adType.name.lowercase())
        )
        originListener?.onAdClicked(molocoAd)
    }
}
