package com.moloco.sdk.internal.publisher

import android.content.Context
import com.moloco.sdk.internal.DefaultMolocoFullscreenAdAggregatedOptions
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.SdkEventUrlTracker
import com.moloco.sdk.internal.ortb.model.SdkEvents
import com.moloco.sdk.internal.services.AnalyticsApplicationLifecycleTracker
import com.moloco.sdk.internal.toFullscreenAggregatedOptions
import com.moloco.sdk.publisher.AdFormatType
import com.moloco.sdk.publisher.AdShowListener
import com.moloco.sdk.publisher.MolocoAd
import com.moloco.sdk.publisher.MolocoAdError
import com.moloco.sdk.publisher.RewardedInterstitialAd
import com.moloco.sdk.publisher.RewardedInterstitialAdShowListener
import com.moloco.sdk.publisher.createAdInfo
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.CreativeType
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.Watermark
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ExternalLinkHandler
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService
import com.moloco.sdk.xenoss.sdkdevkit.android.persistenttransport.PersistentHttpRequest

/**
 * @param adDataHolder is only visible for it to be mocked by unit tests
 */
internal fun RewardedInterstitialAd(
        context: Context,
        appLifecycleTrackerService: AnalyticsApplicationLifecycleTracker,
        customUserEventBuilderService: CustomUserEventBuilderService,
        adUnitId: String,
        externalLinkHandler: ExternalLinkHandler,
        persistentHttpRequest: PersistentHttpRequest,
        adDataHolder: FullscreenAdDataHolder<RewardedInterstitialAdShowListener> =
                FullscreenAdDataHolder(),
        watermark: Watermark,
        adCreateLoadTimeoutManager: AdCreateLoadTimeoutManager
): RewardedInterstitialAd = RewardedInterstitialAdImpl(
    FullscreenAdImpl(
            context,
            appLifecycleTrackerService,
            customUserEventBuilderService,
            adUnitId,
            persistentHttpRequest = persistentHttpRequest,
            externalLinkHandler,
            generateAggregatedOptions = { playerExt ->
                playerExt?.toFullscreenAggregatedOptions()
                    ?: DefaultMolocoFullscreenAdAggregatedOptions()
            },
            adDataHolder = adDataHolder,
            adFormatType = AdFormatType.REWARDED,
            watermark = watermark,
            adCreateLoadTimeoutManager = adCreateLoadTimeoutManager
    ),
    adUnitId
)

private class RewardedInterstitialAdImpl(
        private val fullscreenAd: FullscreenAdImpl<RewardedInterstitialAdShowListener>,
        private val adUnitId: String,
) :
    RewardedInterstitialAd,
    CreateAdObjectTime by fullscreenAd,
    com.moloco.sdk.publisher.FullscreenAd<RewardedInterstitialAdShowListener> by fullscreenAd {

    override fun show(listener: RewardedInterstitialAdShowListener?) {
        // Tracker wrapping.
        val listenerTracker = RewardedInterstitialAdShowListenerTracker(
            originListener = listener,
            provideSdkEvents = { fullscreenAd.sdkEvents },
        )

        val finalListener =
            createRewardedListenerFromInterstitialListener(
                listenerTracker,
                isVast = fullscreenAd.creativeType == CreativeType.VAST,
                isAdForciblyClosed = { fullscreenAd.isAdForciblyClosed }
            )

        fullscreenAd.vastCompletionStatusListener = {
            finalListener.onRewardedVideoCompleted(createAdInfo(adUnitId))
        }

        fullscreenAd.show(finalListener)
    }
}

/**
 * Transforms interstitial callbacks into rewarded ones, since they are not supported/called by ::FullscreenAd.
 */
internal fun createRewardedListenerFromInterstitialListener(
    listenerTracker: RewardedInterstitialAdShowListener,
    isVast: Boolean,
    isAdForciblyClosed: () -> Boolean?
) = object : RewardedInterstitialAdShowListener {

    private val TAG = "RewardedInterstitialAdShowListenerImpl"

    private var isRewardHandled = false

    /**
     * Issue reward when:
     * - VAST video part completes
     * - VAST video part is skipped
     * - Standalone STATIC HTML or MRAID reward ad; VAST endcard, VAST video endframe is closed.
     */
    private fun tryIssueReward(molocoAd: MolocoAd) {
        if (isRewardHandled) {
            MolocoLogger.debug(TAG, "issuing of reward is already handled")
            return
        }

        isRewardHandled = true

        // if ad is forcibly closed then we do not give reward.
        val issueReward = isAdForciblyClosed() == false
        if (issueReward) {
            MolocoLogger.debug(TAG, "issuing of reward...")
            onUserRewarded(molocoAd)
        } else {
            MolocoLogger.debug(TAG, "reward can't be issued: ad was forcibly closed or ad was missing")
        }
    }

    override fun onUserRewarded(molocoAd: MolocoAd) {
        listenerTracker.onUserRewarded(molocoAd)
    }

    override fun onRewardedVideoStarted(molocoAd: MolocoAd) {
        listenerTracker.onRewardedVideoStarted(molocoAd)
    }

    override fun onRewardedVideoCompleted(molocoAd: MolocoAd) {
        // Issuing rewards on vast video ad part skip or video playback completion.
        tryIssueReward(molocoAd)
        listenerTracker.onRewardedVideoCompleted(molocoAd)
    }

    override fun onAdShowSuccess(molocoAd: MolocoAd) {
        listenerTracker.onAdShowSuccess(molocoAd)
        if (isVast) onRewardedVideoStarted(molocoAd)
    }

    override fun onAdShowFailed(molocoAdError: MolocoAdError) {
        listenerTracker.onAdShowFailed(molocoAdError)
    }

    override fun onAdHidden(molocoAd: MolocoAd) {
        // Issuing rewards on ad close, regardless of type (STATIC, MRAID, VAST etc).
        tryIssueReward(molocoAd)
        listenerTracker.onAdHidden(molocoAd)
    }

    override fun onAdClicked(molocoAd: MolocoAd) {
        listenerTracker.onAdClicked(molocoAd)
    }
}

internal fun RewardedInterstitialAdShowListenerTracker(
    originListener: RewardedInterstitialAdShowListener?,
    provideSdkEvents: () -> SdkEvents?,
): RewardedInterstitialAdShowListener = RewardedInterstitialAdShowListenerTrackerImpl(
    originListener,
    provideSdkEvents,
    SdkEventUrlTracker(),
)

private class RewardedInterstitialAdShowListenerTrackerImpl(
    private val originListener: RewardedInterstitialAdShowListener?,
    private val provideSdkEvents: () -> SdkEvents?,
    private val sdkEventUrlTracker: SdkEventUrlTracker,
) : RewardedInterstitialAdShowListener,
    AdShowListener by AdShowListenerTracker(
        originListener
    ) {

    override fun onAdShowFailed(molocoAdError: MolocoAdError) {
        originListener?.onAdShowFailed(molocoAdError)
    }

    override fun onUserRewarded(molocoAd: MolocoAd) {
        provideSdkEvents()?.onUserRewarded?.let { url ->
            sdkEventUrlTracker(url, System.currentTimeMillis())
        }

        originListener?.onUserRewarded(molocoAd)
    }

    override fun onRewardedVideoStarted(molocoAd: MolocoAd) {
        provideSdkEvents()?.onRewardedVideoStarted?.let { url ->
            sdkEventUrlTracker(url, System.currentTimeMillis())
        }

        originListener?.onRewardedVideoStarted(molocoAd)
    }

    override fun onRewardedVideoCompleted(molocoAd: MolocoAd) {
        provideSdkEvents()?.onRewardedVideoCompleted?.let { url ->
            sdkEventUrlTracker(url, System.currentTimeMillis())
        }

        originListener?.onRewardedVideoCompleted(molocoAd)
    }
}
