package com.moloco.sdk.internal.services

import android.content.ContentValues.TAG
import android.content.SharedPreferences
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import kotlinx.coroutines.withContext
import java.lang.ClassCastException

/**
 * SDK wide key/value data store that provides type safe reads/writes
 */
internal interface DataStoreService {
    suspend fun getInt(key: String): Int?
    suspend fun getString(key: String): String?
    suspend fun getBoolean(key: String): Boolean?
    suspend fun getFloat(key: String): Float?
    suspend fun getDouble(key: String): Double?
    suspend fun getLong(key: String): Long?
    suspend fun removeKey(key: String)

    suspend fun <T> set(key: String, newValue: T)
}

internal class PreferencesDataStoreServiceImpl(private val sharedPreferences: SharedPreferences) : DataStoreService {
    override suspend fun getInt(key: String): Int? = withContext(DispatcherProvider().io) {
        if (!sharedPreferences.contains(key)) {
            return@withContext null
        }

        val value = try {
            sharedPreferences.getInt(key, 0)
        } catch (e: ClassCastException) {
            MolocoLogger.warn(TAG, "Unexpected value type for key: $key")
            null
        }

        return@withContext value
    }

    override suspend fun getString(key: String): String? = withContext(DispatcherProvider().io) {
        if (!sharedPreferences.contains(key)) {
            return@withContext null
        }

        val value = try {
            sharedPreferences.getString(key, "")
        } catch (e: ClassCastException) {
            MolocoLogger.warn(TAG, "Unexpected value type for key: $key")
            null
        }

        return@withContext value
    }

    override suspend fun getBoolean(key: String): Boolean? = withContext(DispatcherProvider().io) {
        if (!sharedPreferences.contains(key)) {
            return@withContext null
        }

        val value = try {
            sharedPreferences.getBoolean(key, false)
        } catch (e: ClassCastException) {
            MolocoLogger.warn(TAG, "Unexpected value type for key: $key")
            null
        }

        return@withContext value
    }

    override suspend fun getFloat(key: String): Float? = withContext(DispatcherProvider().io) {
        if (!sharedPreferences.contains(key)) {
            return@withContext null
        }

        val value = try {
            sharedPreferences.getFloat(key, 0f)
        } catch (e: ClassCastException) {
            MolocoLogger.warn(TAG, "Unexpected value type for key: $key")
            null
        }

        return@withContext value
    }

    override suspend fun getDouble(key: String): Double? = withContext(DispatcherProvider().io) {
        if (!sharedPreferences.contains(key)) {
            return@withContext null
        }

        val value = try {
            sharedPreferences.getString(key, "0")?.toDouble()
        } catch (ce: ClassCastException) {
            MolocoLogger.warn(TAG, "Unexpected value type for key: $key", ce)
            null
        } catch (nfe: NumberFormatException) {
            MolocoLogger.warn(TAG, "Unexpected value type for key: $key", nfe)
            null
        }

        return@withContext value
    }

    override suspend fun getLong(key: String): Long? = withContext(DispatcherProvider().io) {
        if (!sharedPreferences.contains(key)) {
            return@withContext null
        }

        val value = try {
            sharedPreferences.getString(key, "0")?.toLong()
        } catch (ce: ClassCastException) {
            MolocoLogger.warn(TAG, "Unexpected value type for key: $key", ce)
            null
        } catch (nfe: NumberFormatException) {
            MolocoLogger.warn(TAG, "Unexpected value type for key: $key", nfe)
            null
        }

        return@withContext value
    }

    override suspend fun removeKey(key: String) = withContext(DispatcherProvider().io) {
        sharedPreferences.edit().remove(key).apply()
    }

    override suspend fun <T> set(key: String, newValue: T) = withContext(DispatcherProvider().io) {
        when (newValue) {
            is Int -> sharedPreferences.edit().putInt(key, newValue).apply()
            is String -> sharedPreferences.edit().putString(key, newValue).apply()
            is Float -> sharedPreferences.edit().putFloat(key, newValue).apply()
            is Boolean -> sharedPreferences.edit().putBoolean(key, newValue).apply()
            is Double -> sharedPreferences.edit().putString(key, newValue.toString()).apply()
            is Long -> sharedPreferences.edit().putString(key, newValue.toString()).apply()
            else -> {
                MolocoLogger.warn(TAG, "Unexpected value type: $newValue for key: $key")
            }
        }
    }
}
