package com.moloco.sdk.internal.services.bidtoken

import com.moloco.sdk.acm.AndroidClientMetrics
import com.moloco.sdk.acm.CountEvent
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.client_metrics_data.AcmCount
import com.moloco.sdk.internal.client_metrics_data.AcmResultTag
import com.moloco.sdk.internal.client_metrics_data.AcmTag
import com.moloco.sdk.internal.publisher.InitializationHandler
import com.moloco.sdk.internal.services.TimeProviderService
import com.moloco.sdk.publisher.Initialization
import com.moloco.sdk.publisher.Moloco
import com.moloco.sdk.publisher.MolocoAdError
import com.moloco.sdk.publisher.MolocoBidTokenListener
import java.util.concurrent.TimeUnit

internal interface BidTokenHandler {
    /**
     * Fetches the bid token and returns it to the listener
     */
    suspend fun handleBidTokenRequest(listener: MolocoBidTokenListener)
}

internal class BidTokenHandlerImpl(
    private val bidTokenService: BidTokenService,
    private val initializationHandler: InitializationHandler,
    private val timeProviderService: TimeProviderService,
    private val acm: AndroidClientMetrics,
) : BidTokenHandler {
    private val TAG = "BidTokenHandlerImpl"

    override suspend fun handleBidTokenRequest(listener: MolocoBidTokenListener) {
        acm.recordCountEvent(CountEvent(AcmCount.BidTokenGetRequest.eventName))
        // If the SDK cannot initialize at all, don't allow making bid request that will eventually fail
        if (initializationHandler.canInitialize().not()) {
            val error = MolocoAdError.ErrorType.SDK_PERSISTENT_HTTP_REQUEST_FAILED_TO_INIT
            MolocoLogger.info(TAG, "Bid token cannot be fetched because SDK initialization cannot happen due to WM issue")
            acm.recordCountEvent(CountEvent(AcmCount.BidTokenGetResponse.eventName)
                .withTag(AcmTag.Result.tagName, AcmResultTag.failure.name)
                .withTag(AcmTag.Reason.tagName, "sdk_cannot_initialize")
            )
            listener.onBidTokenResult("", error)
            return
        }

        // If SDK initialization has failed, don't allow making bid request by sending failed bid token
        // MAX does not check for Moloco SDK initialization status and keeps sending us the sdk ad load request
        if (initializationHandler.initializationState.value == Initialization.FAILURE) {
            MolocoLogger.info(TAG, "Bid token cannot be fetched because SDK initialization has failed")
            acm.recordCountEvent(CountEvent(AcmCount.BidTokenGetResponse.eventName)
                .withTag(AcmTag.Result.tagName, AcmResultTag.failure.name)
                .withTag(AcmTag.Reason.tagName, "sdk_init_failed")
            )
            listener.onBidTokenResult("", MolocoAdError.ErrorType.SDK_INIT_ERROR)
            return
        }

        val bidTokenLoadStart = timeProviderService.currentTime()
        val bidToken = bidTokenService.bidToken()
        val bidTokenLoadDuration = timeProviderService.currentTime() - bidTokenLoadStart

        MolocoLogger.info(TAG, "Bid token fetched in $bidTokenLoadDuration ms")

        val error = if (bidToken.isEmpty()) {
            acm.recordCountEvent(CountEvent(AcmCount.BidTokenGetResponse.eventName)
                .withTag(AcmTag.Result.tagName, AcmResultTag.failure.name)
                .withTag(AcmTag.Reason.tagName, "bid_token_fetch_failed")
            )

            if (bidTokenLoadDuration >= TimeUnit.SECONDS.toMillis(1)) {
                acm.recordCountEvent(CountEvent(AcmCount.BidTokenDurationTimeoutOneSecond.eventName)
                    .withTag(AcmTag.Result.tagName, AcmResultTag.failure.name)
                    .withTag(AcmTag.Reason.tagName, "bid_token_fetch_failed"))
            }

            if (bidTokenLoadDuration >= TimeUnit.SECONDS.toMillis(3)) {
                acm.recordCountEvent(CountEvent(AcmCount.BidTokenDurationTimeoutThreeSecond.eventName)
                    .withTag(AcmTag.Result.tagName, AcmResultTag.failure.name)
                    .withTag(AcmTag.Reason.tagName, "bid_token_fetch_failed"))
            }

            MolocoAdError.ErrorType.AD_SIGNAL_COLLECTION_FAILED
        } else {
            acm.recordCountEvent(CountEvent(AcmCount.BidTokenGetResponse.eventName)
                .withTag(AcmTag.Result.tagName, AcmResultTag.success.name)
            )

            if (bidTokenLoadDuration >= TimeUnit.SECONDS.toMillis(1)) {
                acm.recordCountEvent(CountEvent(AcmCount.BidTokenDurationTimeoutOneSecond.eventName)
                    .withTag(AcmTag.Result.tagName, AcmResultTag.success.name))
            }

            if (bidTokenLoadDuration >= TimeUnit.SECONDS.toMillis(3)) {
                acm.recordCountEvent(CountEvent(AcmCount.BidTokenDurationTimeoutThreeSecond.eventName)
                    .withTag(AcmTag.Result.tagName, AcmResultTag.success.name))
            }

            null
        }



        MolocoLogger.info(TAG, "Returning bid token result, hasError: ${error != null}, SDK init complete: ${Moloco.isInitialized}")
        listener.onBidTokenResult(bidToken, error)
    }
}
