package com.moloco.sdk.internal.services.bidtoken

import android.os.Build
import androidx.annotation.VisibleForTesting
import com.moloco.sdk.BuildConfig
import com.moloco.sdk.acm.AndroidClientMetrics
import com.moloco.sdk.acm.CountEvent
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.Result
import com.moloco.sdk.internal.bidtoken.VersionPrefixedJWTokenParser
import com.moloco.sdk.internal.client_metrics_data.AcmCount
import com.moloco.sdk.internal.client_metrics_data.AcmTag
import com.moloco.sdk.internal.client_metrics_data.AcmTimer
import com.moloco.sdk.internal.http.buildHttpClient
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.internal.services.TimeProviderServiceImpl
import com.moloco.sdk.internal.utils.withReentrantLock
import com.moloco.sdk.service_locator.SdkObjectFactory
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Job
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.launch
import kotlinx.coroutines.sync.Mutex
import java.util.Locale
import com.moloco.sdk.acm.AndroidClientMetrics as acm

internal typealias BidTokenFetchResult = BidTokenResponseComponents
internal interface ServerBidTokenService {
    companion object {
        fun create(): ServerBidTokenService = Instance
    }

    suspend fun bidToken(): BidTokenFetchResult
}

/**
 * A stateful service that manages the BidToken at the SDK layer. The BidToken
 * can expire based on numerous factors and this service is responsible for
 * refreshing the token when necessary.
 */
private val Instance by lazy {
    MolocoLogger.info("ServerBidTokenService", "Creating BidTokenService instance")
    ServerBidTokenServiceImpl(
        bidTokenApi = BidTokenApiImpl(
            sdkVersion = BuildConfig.SDK_VERSION_NAME,
            // We create a different http client here because `Moloco` SDK could not be initialized yet and we
            // don't want the factory to cache the http client with null app key httpclient
            httpClient = buildHttpClient(
                appInfo = SdkObjectFactory.DeviceAndApplicationInfo.appInfoSingleton(),
                deviceInfo = SdkObjectFactory.DeviceAndApplicationInfo.deviceInfoSingleton(),
            ),
            httpRequestInfo = BidTokenHttpRequestInfo(
                requestTimeoutMillis = 2800,
                fetchRetryMax = 3,
                fetchRetryDelayMillis = 200, // MAX has a timeout of 3s for the bid token
            ),
            deviceRequestInfo = BidTokenDeviceRequestInfo(
                language = Locale.getDefault().language,
                osVersion = Build.VERSION.RELEASE,
                make = Build.MANUFACTURER ?: "",
                model = Build.MODEL ?: "",
                hardwareVersion = Build.HARDWARE ?: "",
            ),
        ),
        scope = CoroutineScope(SupervisorJob() + DispatcherProvider().io),
        tokenCache = ServerBidTokenCache.create(
            bidTokenParser = VersionPrefixedJWTokenParser(),
            timeProviderService = TimeProviderServiceImpl(),
        ),
    )
}

@VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
internal class ServerBidTokenServiceImpl(
    private val bidTokenApi: BidTokenApi,
    private val scope: CoroutineScope,
    private val tokenCache: ServerBidTokenCache,
) : ServerBidTokenService {
    private val TAG = "ServerBidTokenServiceImpl"

    /**
     * This flag is used to determine if the bid token has been fetched at least once in the same app session
     * This is used for analytics purposes
     */
    private var initialFetch = true
    private val mutex = Mutex()
    private var asyncRefreshJob: Job? = null

    /**
     * Refreshes the bid token asynchronously. If a refresh is already in progress, this function
     * throttles the duplicate one.
     */
    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    internal fun refreshTokenAsyncOnExpiry() {
        infoLog( "[Thread: ${Thread.currentThread().name}] Refreshing token async")
        acm.recordCountEvent(CountEvent(AcmCount.ServerBidTokenAsyncRefresh.eventName)
            .withTag("async", (asyncRefreshJob?.isActive ?: false).toString())
        )
        if (asyncRefreshJob?.isActive == true) {
            infoLog( "[Thread: ${Thread.currentThread().name}] Async refresh already in progress. Returning")
            return
        }

        infoLog( "[Thread: ${Thread.currentThread().name}] Scheduling to fetch token from server")
        asyncRefreshJob = scope.launch {
            infoLog( "[Thread: ${Thread.currentThread().name}] Fetching token from server")
            fetchServerBidToken(DefaultBidTokenResponseComponents, asyncFetch = true, wasExpiring = true)
            infoLog( "[Thread: ${Thread.currentThread().name}] Finished fetching token from server")
        }
    }

    /**
     * Fetches the bid token from the server from the calling coroutine. This function updates
     * the token cache *ONLY* when the API fetch is successful and returns the new token.
     * In case of a failure, the cache is not updated and the result is returned as is.
     *
     * @param defaultTokenOnError The default token to return in case of a failure
     * @param asyncFetch Whether the fetch is from async refresh or not (For analytic purposes)
     * @param wasExpiring Whether the token was expiring or not (For analytic purposes)
     */
    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    internal suspend fun fetchServerBidToken(defaultTokenOnError: BidTokenResponseComponents,
                                             asyncFetch :Boolean,
                                             wasExpiring: Boolean
    ): BidTokenFetchResult {
        val bidTokenTimer = acm.startTimerEvent(AcmTimer.ServerBidTokenFetch.eventName)
        infoLog( "[Thread: ${Thread.currentThread().name}] fetchServerBidToken")
        val bidTokenComponents =
        when (val bidTokenFetchResult = bidTokenApi.fetchBidToken()) {
            is Result.Failure -> {
                acm.recordCountEvent(CountEvent(AcmCount.ServerBidTokenFetch.eventName)
                    .withTag(AcmTag.Result.tagName, "failure")
                    .withTag(AcmTag.Reason.tagName, "${bidTokenFetchResult.value.failureCode}")
                    .withTag("initial_fetch", initialFetch.toString())
                    .withTag("was_expiring", wasExpiring.toString())
                    .withTag("async", asyncFetch.toString())
                )
                acm.recordTimerEvent(bidTokenTimer
                    .withTag(AcmTag.Result.tagName, "failure")
                    .withTag(AcmTag.Reason.tagName, "${bidTokenFetchResult.value.failureCode}")
                    .withTag("initial_fetch", initialFetch.toString())
                    .withTag("was_expiring", wasExpiring.toString())
                    .withTag("async", asyncFetch.toString())
                )
                MolocoLogger.error(
                    TAG,
                    "bidtoken request failed: ${bidTokenFetchResult.value.failureCode}, details: ${bidTokenFetchResult.value.description}"
                )
                defaultTokenOnError
            }

            is Result.Success -> {
                acm.recordCountEvent(CountEvent(AcmCount.ServerBidTokenFetch.eventName)
                    .withTag(AcmTag.Result.tagName, "success")
                    .withTag("initial_fetch", initialFetch.toString())
                    .withTag("was_expiring", wasExpiring.toString())
                    .withTag("async", asyncFetch.toString())
                )
                acm.recordTimerEvent(bidTokenTimer
                    .withTag(AcmTag.Result.tagName, "success")
                    .withTag("initial_fetch", initialFetch.toString())
                    .withTag("was_expiring", wasExpiring.toString())
                    .withTag("async", asyncFetch.toString())
                )

                AndroidClientMetrics.recordCountEvent(
                    CountEvent(AcmCount.ServerBidTokenCached.eventName)
                        .withTag(AcmTag.Result.tagName, "false")
                        .withTag("initial_fetch", initialFetch.toString())
                        .withTag("async", asyncFetch.toString())
                )
                infoLog( "[Thread: ${Thread.currentThread().name}] bidtoken request success")
                mutex.withReentrantLock {
                    tokenCache.updateCache(bidTokenFetchResult.value)
                }

                bidTokenFetchResult.value.also {
                    initialFetch = false
                }
            }
        }

        return bidTokenComponents
    }

    override suspend fun bidToken(): BidTokenFetchResult {
        infoLog( "[Thread: ${Thread.currentThread().name}] Fetching bidToken(), acquiring lock")
        return mutex.withReentrantLock {
            infoLog( "[Thread: ${Thread.currentThread().name}] Acquired lock, fetching status of current token")
            val refreshStatus = tokenCache.tokenStatus()
            debugBuildLog( "[Thread: ${Thread.currentThread().name}] bidToken status: $refreshStatus")
            if (refreshStatus.canUseToken()) {
                AndroidClientMetrics.recordCountEvent(
                    CountEvent(AcmCount.ServerBidTokenCached.eventName)
                        .withTag(AcmTag.Result.tagName, "true")
                        .withTag("initial_fetch", false.toString())
                        .withTag("expiring", if (refreshStatus == ServerTokenCacheStatus.EXPIRING) "true" else "false")
                )
                val cachedToken = tokenCache.cachedToken
                if (refreshStatus == ServerTokenCacheStatus.EXPIRING) {
                    debugBuildLog( "[Thread: ${Thread.currentThread().name}] bidToken is expiring, returning cached, and refreshing async")
                    refreshTokenAsyncOnExpiry()
                } else {
                    debugBuildLog( "[Thread: ${Thread.currentThread().name}] bidToken doesn't need refresh, returning cached")
                }
                return@withReentrantLock cachedToken
            }

            // Expired or does not exist
            infoLog( "[Thread: ${Thread.currentThread().name}] bidToken needs refresh, fetching new token")

            // NOTE: It is possible that there is an active async job fetching the token, while the current token expires. This should be a rare
            // enough scenario in which case let's just fetch the token again instead of maintaining a complex logic of waiting on the refresh job
            // and handling all the thread safety related issues.
            return@withReentrantLock fetchServerBidToken(DefaultBidTokenResponseComponents, asyncFetch = false, wasExpiring = false)
        }
    }

    private fun infoLog(message: String) {
        MolocoLogger.info(TAG, "[Thread: ${Thread.currentThread().name}][sbt] $message")
    }

    private fun debugBuildLog(message: String) {
        MolocoLogger.debugBuildLog(TAG, "[Thread: ${Thread.currentThread().name}] $message")
    }
}


