package com.moloco.sdk.internal.services.bidtoken.providers

import android.content.Context
import com.moloco.sdk.internal.MolocoLogger

/**
 * A signal provider that provides information on application directory
 */
internal data class AppDirInfo(
    /**
     * Size of the application directory in bytes
     */
    val appDirSize: Long?
)

internal class AppDirInfoSignalProvider(private val context: Context): ClientBidTokenSignalProvider<AppDirInfo> {
    private var cachedAppDirInfo = AppDirInfo(getAppTotalDiskSize())

    override fun needsRefresh(): Boolean {
        val currentAppDirInfo = AppDirInfo(getAppTotalDiskSize())
        val needsRefresh = currentAppDirInfo != cachedAppDirInfo
        MolocoLogger.debugBuildLog(TAG, "[CBT] ADI needsRefresh: $needsRefresh, with adi: ${currentAppDirInfo.appDirSize}")
        return needsRefresh
    }

    override fun tryUpdateSignalState() {
        cachedAppDirInfo = AppDirInfo(getAppTotalDiskSize())
    }

    override fun provideSignal(): AppDirInfo {
        return cachedAppDirInfo.apply {
            MolocoLogger.debugBuildLog(TAG, "[CBT] ADI providing ${this.appDirSize}")
        }
    }

    override fun logTag(): String = TAG

    // Got same results when used the context.filesDir and StatFs
    // 1) Using context.filesDir
    // 12-09 16:12:00.632 29003 29100 D MolocoADISignalProvider: [getTotalDiskSize] [CBT] filesDir: /data/user/0/com.trickytribe.penetrator/files, size: 235860361216
    // Using StatFs
    // 2) 12-09 16:17:01.825 29813 29913 D MolocoADISignalProvider: [getTotalDiskSize] [CBT] totalSpace: 235860361216, freeSpace: 220770869248, usedSpace: 15089491968
    private fun getAppTotalDiskSize(): Long? {
        return try {
            context.filesDir.totalSpace
        } catch (e: Exception) {
            // SecurityException can be thrown, but in general we catch all exceptions and handle that gracefully
            MolocoLogger.error(TAG, "ADI Error", e)
            null
        }
    }

    companion object {
        private const val TAG = "ADISignalProvider"
    }
}
