package com.moloco.sdk.internal.services.bidtoken.providers

import android.app.ActivityManager
import com.google.common.annotations.VisibleForTesting
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.services.AdData
import com.moloco.sdk.internal.services.bidtoken.MolocoPrivacyBasedPrivacyProvider
import com.moloco.sdk.publisher.privacy.MolocoPrivacy
import com.moloco.sdk.service_locator.SdkObjectFactory

/**
 * A collection of client signals that we pass to our BidToken service
 */
internal data class ClientSignals(
    val sdkInitialized: Boolean,
    val privacySettings: MolocoPrivacy.PrivacySettings,
    val memoryInfo: MemoryInfoSignal,
    val appDirInfo: AppDirInfo,
    val networkInfoSignal: NetworkInfoSignal,
    val batteryInfoSignal: BatteryInfoSignal,
    val adDataSignal: AdData,
    val deviceSignal: DeviceSignalInfo,
    val audioSignal: AudioSignal,
    val accessibilitySignal: AccessibilitySignal,
)

/**
 * A composite signal provider that aggregates signals from multiple signal providers
 */
internal interface CompositeClientBidTokenSignalProvider : ClientBidTokenSignalProvider<ClientSignals> {
    companion object {

        /**
         * Create a composite signal provider with the required set of signal providers
         */
        fun create(): CompositeClientBidTokenSignalProvider =
            CompositeClientBidTokenSignalProviderImpl(listOf(
                SDKInitStateSignalProvider(),
                PrivacyStateSignalProvider(MolocoPrivacyBasedPrivacyProvider()),
                MemorySignalProvider(SdkObjectFactory.Miscellaneous.activityManagerSingleton),
                AppDirInfoSignalProvider(SdkObjectFactory.context),
                NetworkInfoSignalProvider(SdkObjectFactory.Network.networkInfoSingleton),
                BatteryInfoSignalProvider(SdkObjectFactory.DeviceAndApplicationInfo.batteryInfoSingleton),
                AdvertisingSignalProvider(SdkObjectFactory.Miscellaneous.adDataSingleton),
                DeviceSignalProvider(SdkObjectFactory.DeviceAndApplicationInfo.deviceInfoSingleton,
                    SdkObjectFactory.DeviceAndApplicationInfo.screenInfoSingleton),
                AudioSignalProvider(SdkObjectFactory.DeviceAndApplicationInfo.audioSingleton),
                AccessibilitySignalProvider(SdkObjectFactory.DeviceAndApplicationInfo.accessibilityServiceSingleton),
            ))
    }
}

internal class CompositeClientBidTokenSignalProviderImpl(@VisibleForTesting internal val signalProviders: List<ClientBidTokenSignalProvider<*>>) : CompositeClientBidTokenSignalProvider {
    override fun needsRefresh(): Boolean {
        return signalProviders.any {signalProvider ->
            signalProvider.needsRefresh().also {
                if (it) {
                    // Logging in production so we can investigate any prod issues if need be
                    MolocoLogger.info(TAG, "[CBT] Signal provider ${signalProvider.logTag()} needs refresh")
                }
            }
        }
    }

    override fun tryUpdateSignalState() {
        for (provider in signalProviders) {
            provider.tryUpdateSignalState()
        }
    }

    override fun provideSignal(): ClientSignals {
        val providerMap = signalProviders.associateBy { it::class }
        // The provider map should always have the following keys as these are required for the bid token,
        // otherwise this will lead to a runtime crash that should only happen during development.
        // We also have a unit test that ensures that the following signal provides are always included. Please keep the unit test in sync with this list.
        val sdkInitState = (providerMap[SDKInitStateSignalProvider::class] as SDKInitStateSignalProvider).provideSignal()
        val privacySettings = (providerMap[PrivacyStateSignalProvider::class] as PrivacyStateSignalProvider).provideSignal()
        val memoryInfo = (providerMap[MemorySignalProvider::class] as MemorySignalProvider).provideSignal()
        val appDirInfo = (providerMap[AppDirInfoSignalProvider::class] as AppDirInfoSignalProvider).provideSignal()
        val networkInfoSignal = (providerMap[NetworkInfoSignalProvider::class] as NetworkInfoSignalProvider).provideSignal()
        val batteryInfoSignal = (providerMap[BatteryInfoSignalProvider::class] as BatteryInfoSignalProvider).provideSignal()
        val adDataSignal = (providerMap[AdvertisingSignalProvider::class] as AdvertisingSignalProvider).provideSignal()
        val deviceSignal = (providerMap[DeviceSignalProvider::class] as DeviceSignalProvider).provideSignal()
        val audioSignal = (providerMap[AudioSignalProvider::class] as AudioSignalProvider).provideSignal()
        val accessibilitySignal = (providerMap[AccessibilitySignalProvider::class] as AccessibilitySignalProvider).provideSignal()
        return ClientSignals(sdkInitState, privacySettings, memoryInfo, appDirInfo, networkInfoSignal, batteryInfoSignal, adDataSignal, deviceSignal, audioSignal, accessibilitySignal)
    }

    override fun logTag(): String = TAG

    companion object {
        private const val TAG = "ClientBidTokenSignalProviderImpl"
    }
}
