package com.moloco.sdk.internal.services.config

import com.moloco.sdk.Init.SDKInitResponse
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.services.config.handlers.ConfigHandler
import com.moloco.sdk.internal.services.config.handlers.OperationalMetricsConfigHandler
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.media.DefaultMediaConfig
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.media.MediaConfig

/**
 * A configuration service that retrieves configurations from the SDKInitResponse
 * and provides them to the SDK components.
 */
internal class RemoteConfigService : ConfigService {
    private val TAG = "RemoteConfigService"
    private val configs = mutableMapOf<String, Any>()
    private val experimentalFeatureFlags = mutableMapOf<String, String?>()

    // NOTE - Add your config here
    private val handlers = listOf<ConfigHandler>(
        OperationalMetricsConfigHandler()
    )

    override fun initialize(sdkInitResponse: SDKInitResponse) {
        parseConfigs(sdkInitResponse)
        parseExperimentalFeatureFlags(sdkInitResponse)
    }

    @Suppress("UNCHECKED_CAST")
    override fun <T> getConfig(configType: Class<T>, default: T): T {
        MolocoLogger.debug(TAG, "Retrieving config: $configType")
        return configs[configType.name] as T? ?: default
    }

    override fun hasExperimentFeatureFlag(featureFlagName: String): Boolean {
        return experimentalFeatureFlags.containsKey(featureFlagName)
    }

    override fun getExperimentalFeatureFlagValue(featureFlagName: String): String? {
        return experimentalFeatureFlags[featureFlagName]
    }

    /**
     * Parse the configs from the SDKInitResponse and store them in the configs map
     */
    private fun parseConfigs(sdkInitResponse: SDKInitResponse) {
        handlers.forEach { handler ->
            val config = handler.handleConfig(sdkInitResponse)
            configs[handler.getConfigType().name] = config
            MolocoLogger.info(TAG, "Adding config: ${handler.getConfigType().name}")
        }
        configs[MediaConfig::class.java.name] = parseMediaConfig(sdkInitResponse)
        // TODO: Move [CustomUserBuilderConfigService] as a config here in any future refactors
    }

    private fun parseExperimentalFeatureFlags(sdkInitResponse: SDKInitResponse) {
        sdkInitResponse.experimentalFeatureFlagsList.forEach { flag ->
            MolocoLogger.info(TAG, "Adding ExperimentalFeatureFlag: ${flag.name}")
            experimentalFeatureFlags[flag.name] = if (flag.value.isNullOrEmpty()) null else flag.value
        }
    }

    // TODO: Move configs to use Handler pattern: https://mlc.atlassian.net/browse/SDK-2146
    private fun parseMediaConfig(sdkInitResponse: SDKInitResponse): MediaConfig {
        val isStreamingEnabled = sdkInitResponse.experimentalFeatureFlagsList.map { it.name }.contains("ANDROID_STREAMING_ENABLED")
        MolocoLogger.info(TAG, "Adding StreamingEnabled: $isStreamingEnabled")

        val mediaConfig = if (sdkInitResponse.hasConfigs() &&
            sdkInitResponse.configs.hasCommonConfigs() &&
            sdkInitResponse.configs.commonConfigs.hasMediaConfig()
            ) {
                val protoMediaConfig = sdkInitResponse.configs.commonConfigs.mediaConfig
                val chunkSize = if (sdkInitResponse.configs.commonConfigs.mediaConfig.streamingChunkSizeKilobytes.toInt() > 0) {
                    protoMediaConfig.streamingChunkSizeKilobytes.toInt() * 1024
                } else {
                    DefaultMediaConfig.chunkSize
                }

                val minStreamingPlayableDuration = if (sdkInitResponse.configs.commonConfigs.mediaConfig.minStreamingPlayableDurationOnTimeoutSecs > 0.0) {
                    protoMediaConfig.minStreamingPlayableDurationOnTimeoutSecs
                } else {
                    DefaultMediaConfig.minStreamingPlayableDurationOnTimeoutSecs
                }


                MediaConfig(
                    isStreamingEnabled = isStreamingEnabled,
                    chunkSize = chunkSize,
                    minStreamingPlayableDurationOnTimeoutSecs = minStreamingPlayableDuration,
                    mediaCacheDiskCleanUpLimit = DefaultMediaConfig.mediaCacheDiskCleanUpLimit) // Not read from remote config
        } else {
            MediaConfig(isStreamingEnabled = isStreamingEnabled,
                chunkSize = DefaultMediaConfig.chunkSize,
                minStreamingPlayableDurationOnTimeoutSecs = DefaultMediaConfig.minStreamingPlayableDurationOnTimeoutSecs,
                mediaCacheDiskCleanUpLimit = DefaultMediaConfig.mediaCacheDiskCleanUpLimit)
        }

        MolocoLogger.debug(TAG, "Parsed and adding MediaConfig: ${mediaConfig.chunkSize}, ${mediaConfig.isStreamingEnabled}, ${mediaConfig.minStreamingPlayableDurationOnTimeoutSecs}, ${mediaConfig.mediaCacheDiskCleanUpLimit} ")
        return mediaConfig
    }
}
