package com.moloco.sdk.internal.services.init

import android.content.SharedPreferences
import android.util.Base64
import androidx.annotation.VisibleForTesting
import com.moloco.sdk.Init.SDKInitResponse
import com.moloco.sdk.acm.AndroidClientMetrics
import com.moloco.sdk.acm.CountEvent
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.client_metrics_data.AcmCount
import com.moloco.sdk.internal.client_metrics_data.AcmResultTag
import com.moloco.sdk.internal.client_metrics_data.AcmTag
import com.moloco.sdk.internal.client_metrics_data.AcmTimer
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import kotlinx.coroutines.withContext
import kotlin.coroutines.CoroutineContext

internal const val CACHE_VERSION = "v0"
data class CacheKey(val appKey: String, val mediation: String) {
    fun toKey(): String {
        return "${appKey}___${mediation}___$CACHE_VERSION"
    }

    fun toOldKeys(): List<String> {
        return listOf(
            // When migrating to a new cache version, add the old cache keys here
            // so clean up can be performed (data migration)
//            "${appKey}___${mediation}___v0",
//            "${appKey}___${mediation}___v1"
        )
    }
}


/**
 * Cache for SDKInitResponse
 */
internal interface InitCache {
    /**
     * Update the cache with the given response for the given cache key
     */
    suspend fun updateCache(cacheKey: CacheKey, response: SDKInitResponse)

    /**
     * Get the cached response for the given cache key
     */
    suspend fun get(cacheKey: CacheKey): SDKInitResponse?

    /**
     * Clear the cache for the given cache key and all previous versions of the cache key
     */
    suspend fun clearCache(cacheKey: CacheKey)

    /**
     * Clear the cache fully
     */
    @VisibleForTesting(otherwise = VisibleForTesting.NONE)
    suspend fun clearAll()

    companion object {
        fun create(sharedPreferences: SharedPreferences): InitCache {
            return InitCacheImpl(sharedPreferences, DispatcherProvider().io, AndroidClientMetrics)
        }
    }
}

internal class InitCacheImpl(
    private val sharedPreferences: SharedPreferences,
    private val ioDispatcherContext: CoroutineContext,
    private val acm: AndroidClientMetrics,
) : InitCache {
    override suspend fun updateCache(cacheKey: CacheKey, response: SDKInitResponse) = withContext(ioDispatcherContext) {
        val cacheWriteTimerEvent = acm.startTimerEvent(AcmTimer.SDKInitCacheWrite.name)
        try {
            MolocoLogger.info(TAG, "Updating cache for mediation: ${cacheKey.mediation}")
            val byteArray = response.toByteArray()
            val encodedString = Base64.encodeToString(byteArray, Base64.DEFAULT)
            val encodingFailure = if (encodedString.isNullOrEmpty()) {
                MolocoLogger.warn(TAG, "Failed to encode SDKInitResponse for mediation: ${cacheKey.mediation}")
                true
            } else {
                false
            }
            val writeSuccess = if (!encodingFailure) {
                sharedPreferences.edit().putString(cacheKey.toKey(), encodedString).commit()
            }
            else {
                false
            }

            if (writeSuccess) {
                MolocoLogger.info(TAG, "Successfully updated cache for mediation: ${cacheKey.mediation}")
                acm.recordTimerEvent(cacheWriteTimerEvent
                    .withTag(AcmTag.Result.name, AcmResultTag.success.name)
                )
                acm.recordCountEvent(CountEvent(AcmCount.SDKInitCacheWrite.name)
                    .withTag(AcmTag.Result.name, AcmResultTag.success.name)
                )
            } else {
                val failureReason = if(encodingFailure) "encoding_failure" else "commit_failure"
                MolocoLogger.warn(TAG, "Failed to update cache for mediation: ${cacheKey.mediation} with error: $failureReason")
                acm.recordTimerEvent(cacheWriteTimerEvent
                    .withTag(AcmTag.Result.name, AcmResultTag.failure.name)
                    .withTag(AcmTag.Reason.name, failureReason)
                )
                acm.recordCountEvent(CountEvent(AcmCount.SDKInitCacheWrite.name)
                    .withTag(AcmTag.Result.name, AcmResultTag.failure.name)
                    .withTag(AcmTag.Reason.name, failureReason)
                )
            }
        } catch (e: Exception) {
            MolocoLogger.warn(TAG, "Failed to update cache for mediation: ${cacheKey.mediation} with exception", e)
            acm.recordTimerEvent(cacheWriteTimerEvent
                .withTag(AcmTag.Result.name, AcmResultTag.failure.name)
                .withTag(AcmTag.Reason.name, e.javaClass.simpleName)
            )
            acm.recordCountEvent(CountEvent(AcmCount.SDKInitCacheWrite.name)
                .withTag(AcmTag.Result.name, AcmResultTag.failure.name)
                .withTag(AcmTag.Reason.name, e.javaClass.simpleName)
            )
        }
    }

    override suspend fun get(cacheKey: CacheKey): SDKInitResponse? = withContext(ioDispatcherContext) {
        val cacheReadTimerEvent = acm.startTimerEvent(AcmTimer.SDKInitCacheRead.name)
        try {
            MolocoLogger.info(TAG, "Reading cache for mediation: ${cacheKey.mediation}")
            val encodedString = sharedPreferences.getString(cacheKey.toKey(), null)
            val sdkInitResponse = encodedString?.let {
                val byteArray = Base64.decode(it, Base64.DEFAULT)
                SDKInitResponse.parseFrom(byteArray)
            }

            if (sdkInitResponse != null) {
                MolocoLogger.info(TAG, "Successfully read cache for mediation: ${cacheKey.mediation}")
                acm.recordTimerEvent(cacheReadTimerEvent
                    .withTag(AcmTag.Result.name, AcmResultTag.success.name)
                )
                acm.recordCountEvent(
                    CountEvent(AcmCount.SDKInitCacheRead.name)
                        .withTag(AcmTag.Result.name, AcmResultTag.success.name)
                )
            } else {
                MolocoLogger.info(TAG, "Failed to read from cache (cache_miss) for mediation: ${cacheKey.mediation}")
                acm.recordTimerEvent(cacheReadTimerEvent
                    .withTag(AcmTag.Result.name, AcmResultTag.failure.name)
                    .withTag(AcmTag.Reason.name, "cache_miss")
                )
                acm.recordCountEvent(
                    CountEvent(AcmCount.SDKInitCacheRead.name)
                        .withTag(AcmTag.Result.name, AcmResultTag.failure.name)
                        .withTag(AcmTag.Reason.name, "cache_miss")
                )
            }

            return@withContext sdkInitResponse
        } catch (e: Exception) {
            MolocoLogger.warn(TAG, "Failed to read cache for mediation: ${cacheKey.mediation} with exception", e)
            acm.recordTimerEvent(cacheReadTimerEvent
                .withTag(AcmTag.Result.name, AcmResultTag.failure.name)
                .withTag(AcmTag.Reason.name, e.javaClass.simpleName)
            )
            acm.recordCountEvent(
                CountEvent(AcmCount.SDKInitCacheRead.name)
                    .withTag(AcmTag.Result.name, AcmResultTag.failure.name)
                    .withTag(AcmTag.Reason.name, e.javaClass.simpleName)
            )
            return@withContext null
        }
    }

    override suspend fun clearCache(cacheKey: CacheKey) = withContext(ioDispatcherContext) {
        val cacheClearTimerEvent = acm.startTimerEvent(AcmTimer.SDKInitCacheClear.name)
        try {
            MolocoLogger.info(TAG, "Clearing cache for mediation: ${cacheKey.mediation}")
            val editor = sharedPreferences.edit()
            // Remove the old versioned cached data
            cleanUpOldVersionedCachedData(cacheKey, editor)
            // Remove the current version cached data
            editor.remove(cacheKey.toKey())
            val cachedCleared = editor.commit()
            if (cachedCleared) {
                MolocoLogger.info(TAG, "Successfully cleared cache for mediation: ${cacheKey.mediation}")
                acm.recordCountEvent(CountEvent(AcmCount.SDKInitCacheClear.name)
                    .withTag(AcmTag.Result.name, AcmResultTag.success.name)
                )
                acm.recordTimerEvent(cacheClearTimerEvent
                    .withTag(AcmTag.Result.name, AcmResultTag.success.name)
                )
            } else {
                MolocoLogger.warn(TAG, "Failed to clear cache for mediation: ${cacheKey.mediation}")
                acm.recordCountEvent(CountEvent(AcmCount.SDKInitCacheClear.name)
                    .withTag(AcmTag.Result.name, AcmResultTag.failure.name)
                    .withTag(AcmTag.Reason.name, "commit_failure")
                )
                acm.recordTimerEvent(cacheClearTimerEvent
                    .withTag(AcmTag.Result.name, AcmResultTag.failure.name)
                    .withTag(AcmTag.Reason.name, "commit_failure")
                )
            }
        } catch (e: Exception) {
            MolocoLogger.warn(TAG, "Failed to clear cache for mediation: ${cacheKey.mediation} with exception", e)
            acm.recordCountEvent(CountEvent(AcmCount.SDKInitCacheClear.name)
                .withTag(AcmTag.Result.name, AcmResultTag.failure.name)
                .withTag(AcmTag.Reason.name, e.javaClass.simpleName)
            )
            acm.recordTimerEvent(cacheClearTimerEvent
                .withTag(AcmTag.Result.name, AcmResultTag.failure.name)
                .withTag(AcmTag.Reason.name, e.javaClass.simpleName)
            )
        }
    }

    override suspend fun clearAll() = withContext(ioDispatcherContext) {
        sharedPreferences.edit().clear().commit()
        Unit
    }

    private fun cleanUpOldVersionedCachedData(cacheKey: CacheKey, editor: SharedPreferences.Editor) {
        for (oldKey in cacheKey.toOldKeys()) {
            editor.remove(oldKey)
        }
    }

    companion object {
        private const val TAG = "InitCacheImpl"
    }
}
