package com.moloco.sdk.publisher

import android.net.Uri
import android.view.View

/**
 * Interface representing a native ad for mediation.
 *
 * Implementations of this interface should provide information about the native ad's assets such as
 * title, description, sponsor text, call to action text, rating, icon URI, main image URI, and video view.
 * These properties should return null in case the ad is not loaded, the asset is not available in the bid response,
 * or not preloaded successfully.
 */
interface NativeAd : AdLoad, Destroyable {

    /**
     * Access to the ad's assets.
     */
    val assets: Assets?

    /**
     * Call this when the adapter's view (or some of the asset views, it's up to you to decide) gets rendered.
     */
    fun handleImpression()

    /**
     * Call this when the adapter's view implementation gets clicked.
     */
    fun handleGeneralAdClick()

    /**
     * Register this listener to be informed of [onImpressionHandled] or [onGeneralClickHandled] events.
     */
    var interactionListener: InteractionListener?

    /**
     * Listener to be informed of [onImpressionHandled] or [onGeneralClickHandled] events.
     */
    interface InteractionListener {
        /**
         * Gets called if the latest [handleImpression] call registered an impression.
         * You can use this for counting impressions on the adapter's side.
         */
        fun onImpressionHandled()

        /**
         * Gets called if the latest [handleGeneralAdClick] call registered a click.
         * You can use this for counting clicks on the adapter's side.
         */
        fun onGeneralClickHandled()
    }

    /**
     * Encapsulates all asset-related properties for the ad.
     *
     * These properties should return null in case the ad is not loaded,
     * the asset is not available in the bid response, or not preloaded successfully
     */
    interface Assets {
        /**
         * Title for the ad.
         */
        val title: String?

        /**
         * Text for the ad's body.
         */
        val description: String?

        /**
         * Advertiser for the ad.
         */
        val sponsorText: String?

        /**
         * Call-to-action text.
         */
        val callToActionText: String?

        /**
         * Rating for the ad.
         */
        val rating: Float?

        /**
         * [Uri] link to the icon of the ad.
         */
        val iconUri: Uri?

        /**
         * View for displaying media content. Works for both Image / Video
         */
        val mediaView: View?

        /**
         * If the main asset is an image, then will return the [Uri] link to the image.
         */
        val mainImageUri: Uri?
    }
}
